/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* PC IO/space resident devices test */
/* Begun by Stig Telfer, API, 6 April 1999 */
/* Devices tested so far - cmos, keyboard, mouse */

#include "lib.h"		/* Digital Debug Monitor */

#include "uilib.h"
#include "platform.h"
#include "northbridge.h"
#include "cmos_rtc.h"
#include "kbd.h"
#include "floppy.h"
#include "rom.h"




/*----------------------------------------------------------------------*/
/* Data and definitions used throughout this module */
/* don't change this enum without changing the matching arrays! */

enum isa_dev {	ISADEV_CMOS, ISADEV_KEYB, ISADEV_MOUSE,
		ISADEV_FLOPPY, ISADEV_FLASH, ISADEV_SERIAL,
		ISADEV_NDEVS };

static String dev2str[ISADEV_NDEVS] = {
	"CMOS/Clock",
	"Keyboard",
	"PS/2 Mouse",
	"Floppy/DMA",
	"ROM integrity",
	"Serial ports",
};

static const String str_unknown = "Unknown";
static const String str_testing = "Testing...";
static const String str_passed  = "Passed.";
static const String str_failed  = "FAILED";


static const String str_title = "ISA bus devices test";

/*----------------------------------------------------------------------*/
/* Interface control functions.  All UI updates are done via here */

static void setstate( int i, const String S )
{
    /* Everything is done relative to two global values, r_app and p_app */
    Point p = p_app;
    int maxrow = r_app.h - 2;

    if ( i >= maxrow )       p.x += r_app.w >> 1;
    p.y += i % maxrow;

    mobo_goto( p );
    printf_dbm( "%13s: %-22s", dev2str[ i ], S );
}

static void initstate( void )
{
    int i;

    /* Draw the user interface */
    mobo_box( r_app, str_title );
    for ( i=0; i < ISADEV_NDEVS; i++ )
	setstate( i, str_unknown );
}

/*----------------------------------------------------------------------*/
/* Device test routines */


/*
 * CMOS/Clock test - write and read, then see if it ticks, then print date
 */
static DBM_STATUS cmos_test( void )
{
    int i;
    int ipl;
    unsigned char orig, mode;
    char tmbuf[20];
    DBM_STATUS status = STATUS_SUCCESS;
    DBM_STATUS sval;
    unsigned long start, end, ticks, etime;
    unsigned skipflag;
    long discrep;

    /* Setup: disable interrupts for the duration of this test */
 
    ipl = swpipl( 7 );

    /*
     * 1: is the CMOS there?
     */
    if ( cmosready() != STATUS_SUCCESS )
    {					/* if not ticking, this will timeout */
	setstate( ISADEV_CMOS, "Not responding" );
	mobo_logf( LOG_CRIT "CMOS: The CMOS clock is not ticking\n" );
	swpipl( ipl );
	return STATUS_FAILURE;
    }


#ifndef CONFIG_SHARK		/* CS20 doesn't support this */
    /*
     * 2: test timer modes
     */
    orig = cmosrb( CM_STATUSA ) & 0xF0;

    /* For each mode the timer can be set in (1->15) */
    for ( i=1; i<16; i++ ) {
        mode = orig | i;
        cmoswb( CM_STATUSA, mode );		/* set new clock tick period */

	sprintf_dbm( tmbuf, "test mode %d", i );
	setstate( ISADEV_CMOS, tmbuf );
	mobo_logf( LOG_INFO "CMOS: Timer mode %d (%d us)...", i, TimerModes[i-1] );


	/* measure timer ticks by the interrupt flag, status reg C */
	cmoswb( CM_STATUSB, cmosrb( CM_STATUSB ) | STATUSB_PER ); 
	start = rpcc();
        skipflag = 0;
	(void) cmosrb( CM_STATUSC );			/* clear ints */
	while ( (cmosrb( CM_STATUSC ) & STATUSC_IRQ) == 0 ) {
                end = rpcc();
                if ( end <= start )     end += 1UL << 32;
                ticks = end - start;
                if ( ticks  >= 1UL << 30 ) {            /* a sec or two */
                        skipflag = 1;
#if 0			/* ahem, Nautilus fails one timer mode */
			status = STATUS_WARNING;
#endif
                        mobo_logf("No tick\n"
				  LOG_WARN "CMOS: mode %d is not ticking\n", i );
                        break;
                }
        }
        if ( skipflag )         continue;

        start = rpcc();
        while ( (cmosrb( CM_STATUSC ) & STATUSC_IRQ) == 0 );
        end = rpcc();

        if ( end <= start )     end += 1UL << 32;       /* wraparound */
        ticks = end - start;

	/* Compare accuracy of timed period with what is expected */
        etime = (primary_impure->CYCLE_CNT * ticks) / 1000000UL;
	discrep = (TimerModes[i-1] * 100) / etime;
	if ( discrep > 95 && discrep < 105 )		/* +/- 5 per cent */
        	mobo_logf( "%d us [OK]\n", etime );
	else	mobo_logf( "%d us [FAIL]\n"
			   LOG_CRIT "CMOS: Timer mode %d is inaccurate\n",
		 	   etime, i );
    }

    /* reset to default value */
    cmoswb( CM_STATUSA, orig | 6 );
#endif			/* CONFIG_SHARK */



    /*
     * 3: repeat the update/date OK test 3 times, to see if update is correct
     */
    for ( i=0; i<3; i++)
    {
	sval = datestamp( tmbuf );	/* Test three - what is the date? */
	if( sval != STATUS_SUCCESS )
	{
	    mobo_alertf("Bad date", "Date not valid:\r%s", tmbuf );
	    mobo_logf( LOG_CRIT "CMOS: Date corrupted: %s\n", tmbuf );
	    setstate( ISADEV_CMOS, "Date/time corrupted" );
	    swpipl( ipl );
	    return STATUS_FAILURE;
	}
	setstate( ISADEV_CMOS, tmbuf );
    }

    swpipl( ipl );

    /* last date/time */ 
    mobo_logf( LOG_INFO "CMOS: System date: %s\n", tmbuf);
    if ( status == STATUS_SUCCESS )
	setstate( ISADEV_CMOS, str_passed );

    return status;
}


/*
 * Keyboard test - perform diagnostic self-test, then read bytes from keyb
 */
static DBM_STATUS keyb_test( void ) 
{
#ifdef CONFIG_SHARK			/* CONFIG_SHARK */

    setstate( ISADEV_KEYB, "Skipped (no dev)" ); 
    return STATUS_SUCCESS;

#else					/* CONFIG_SHARK */

    DBM_STATUS r = kbd_init();		/* in devices/kbd.c */
    String report;			/* if report is made, this is !NULL */

    switch (r) {
	case STATUS_SUCCESS:
	    report = str_passed;
	    break;

	case STATUS_FAILURE:
	    report = "Failed initialisation";
	    break;

	default:
	    report = "Unspecified error";
    }
    setstate( ISADEV_KEYB, report );
    return r;

#endif					/* CONFIG_SHARK */
}


/*
 * Mouse test - perform diagnostic self-test, then read bytes from mouse
 */
static DBM_STATUS mouse_test( void )
{
#if 1
    /* fix for Alan, since the diags seem to be not working for mouse,
     * this is a problem but not the end of the world... */
    setstate( ISADEV_MOUSE, "skipped (thats ok)" );
    mobo_logf( LOG_INFO
	"MOUSE: diagnostics skipped in this release (not a problem)\n");
    return STATUS_SUCCESS;
#else						/* temporary mouse fix */
    String report;
    DBM_STATUS r = mouse_init();		/* in devices/kbd.c */

    switch (r) {
	case STATUS_SUCCESS:
	    return STATUS_SUCCESS;

	default:
	case STATUS_FAILURE:
	    report = "Failed initialisation";
	    break;
    }

    setstate( ISADEV_MOUSE, report );
    return STATUS_FAILURE;
#endif						/* temporary mouse fix */
}



/* attempt at a non-destructive floppy test. */

#define FILE_SIZE (256<<10)		/* 256KB file for test */

static DBM_STATUS floppy_test( void )
{

#ifdef CONFIG_SHARK				/* CONFIG_SHARK */

    setstate( ISADEV_FLOPPY, "Skipped (no dev)" );

#else						/* CONFIG_SHARK */

    DBM_STATUS rval;
    const String fname = "diagfile.dat";
    unsigned *tdata;				/* a scratch buffer for data */
    unsigned i, n;
    unsigned exp, act;
    unsigned seed = rpcc();



    /* determine the file size we can use */
    tdata = malloc( FILE_SIZE );
    if ( tdata == NULL )
    {
        setstate( ISADEV_FLOPPY, "No heap mem" );
	return STATUS_FAILURE;
    }
    n = FILE_SIZE / sizeof( tdata[0] );


    /* fill memory with pseudo-random data */
    srandom( seed );
    for ( i=0; i < n; i++ )		tdata[i] = random( );


    /* perform the write */
    
    setstate( ISADEV_FLOPPY, "Writing..." );
    rval = FileWriteRange( fname, tdata, FILE_SIZE );
    if ( rval == STATUS_FAILURE ) {
        setstate( ISADEV_FLOPPY, "Write failure" );
	return STATUS_FAILURE;
    }


    /* clean the slate */
    for ( i=0; i < n; i++ )             tdata[i] = 0;


    /* read back and compare */

    setstate( ISADEV_FLOPPY, "Reading..." );
    if ( LoadAFile( fname, (char *)tdata ) != FILE_SIZE )
    {
        setstate( ISADEV_FLOPPY, "Read failure" );
	return STATUS_FAILURE;
    }
	
    srandom( seed );			/* regenerate pseudorandom sequence */

    for ( i=0; i < n; i++ ) 
    {
	exp = random( ), act = tdata[i];
	if ( exp == act )	continue;	/* not corrupt */

	setstate( ISADEV_FLOPPY, "Data corruption" );
	mobo_logf(
	    LOG_CRIT "FLOPPY: Data corruption, first occurring at byte %d\n"
	    LOG_CRIT "FLOPPY: Expected 0x%08X, got 0x%08X\n",
		   i * sizeof( tdata[0] ), exp, act );

	return STATUS_FAILURE;
    }

    mobo_logf( LOG_INFO "FLOPPY: read/write test passed.\n" );
    setstate( ISADEV_FLOPPY, str_passed );

#endif						/* CONFIG_SHARK */

    return STATUS_SUCCESS;
}




static DBM_STATUS flash_test( void )
{
    DBM_STATUS sval, status = STATUS_SUCCESS;
    BOOLEAN hdr, body;
    rom_t *R;
    char statebuf[20];

    /* Clean up any previous state */
    rom_free( rom_phys_map );
    rom_phys_map = NULL;

    /* Build a new list of all headers present in the ROM */
    setstate( ISADEV_FLASH, "Scanning contents...");
    sval = rom_index();
    if ( sval != STATUS_SUCCESS )
    {
	setstate( ISADEV_FLASH, "ROM unreadable" );
	mobo_logf( LOG_CRIT "FLASH: ROM is unreadable!\n" );
	mobo_alertf( "ROM read failure", "Couldn't find anything in the ROM" );
	return STATUS_FAILURE;
    }

    /* For each ROM image found, verify the image integrity, if applicable */
    for ( R = rom_phys_map; R != NULL; R=R->next )
    {
	/* Can this image be checksummed? */
	if ( (R->flags & ROM_IMGCSUM) == 0 )
		continue;

	sprintf_dbm( statebuf, "Checking %s", R->name );
	setstate( ISADEV_FLASH, statebuf );
        hdr = rom_hdr_valid( R->hdr );
        body = rom_body_valid( R->hdr, R->body );
        if ( !hdr || !body )
        {
	    R->flags &= ~ROM_IMGVALID;			/* image not valid */
            mobo_logf( LOG_CRIT "FLASH: Image at %dKb (%s) did not verify\n",
				R->rstart >> 10, R->name );
            mobo_alertf( "ROM corruption found during scan",
                         "This image (%s at %dKb) appears to be corrupt",
                         R->name, R->rstart >> 10 );
	    status = STATUS_FAILURE;
        }

	rom_dump( R );
    }

    if ( status == STATUS_SUCCESS )
	setstate( ISADEV_FLASH, str_passed );
    else
	setstate( ISADEV_FLASH, "ROM Corruption" );
    return status;
}



#define BYTES_XFER 50000	/* bytes to send in each phase of test */

static DBM_STATUS test_xfer( io_dev xmit, io_dev recv )
{
    unsigned nb, xval, rval;
    srandom( rpcc() );                  /* setup randomness */

    for ( nb=0; nb < BYTES_XFER; nb++ )
    {
        xval = random() & 0xFF;
        io_dev_putc( xmit, xval );
        rval = io_dev_getc_t( recv, 100 );

	if ( xval != rval ) 		return STATUS_FAILURE;
    }
    return STATUS_SUCCESS;
}

static DBM_STATUS serial_test( void )
{
    io_grp tty, log;
    unsigned rval;

    /* Phase 1: initialise COM1 and COM2, if available */
    tty = getty(), log = getlog();
    if ( tty == GRP_COM1 || log == GRP_COM1 ||
#ifdef CONFIG_SHARK
	 tty == GRP_SROM || log == GRP_SROM ||
#endif
	 tty == GRP_COM2 || log == GRP_COM2 )
    {
	mobo_logf( LOG_WARN "SERIAL: serial port already in use, skipping\n");
	setstate( ISADEV_SERIAL, "already in use" );
	return STATUS_WARNING;
    }
	
    if ( io_dev_init( DEV_COM1 ) != STATUS_SUCCESS ||
         io_dev_init( DEV_COM2 ) != STATUS_SUCCESS )
    {
	mobo_logf( LOG_WARN "SERIAL: could not setup ports, skipping test");
	return STATUS_WARNING;
    }


    /* check the loopback cable is in place */
#define RANDOM_VALUE 0xBB
    io_dev_putc( DEV_COM1, RANDOM_VALUE );
    rval = io_dev_getc_t( DEV_COM2, 100 );
    if ( rval == -1 ) {
	mobo_logf( LOG_CRIT "SERIAL: failed transfer - no cable?\n");
	setstate( ISADEV_SERIAL, "no link" );
	return STATUS_WARNING;
    }
    if ( rval != RANDOM_VALUE ) {
	mobo_logf( LOG_CRIT "SERIAL: data corruption during transmission\n");
	setstate( ISADEV_SERIAL, "failed" );
	return STATUS_FAILURE;
    }
#undef RANDOM_VALUE

    srandom( rpcc() );			/* setup randomness */

    /* Phase 2: send from COM1 -> COM2 */
    setstate( ISADEV_SERIAL, "COM1->COM2" );
    if ( test_xfer( DEV_COM1, DEV_COM2 ) != STATUS_SUCCESS ) 
    {
	mobo_logf( LOG_CRIT "SERIAL: error in transfer from COM1 to COM2\n");
	setstate( ISADEV_SERIAL, "failed" );
	return STATUS_FAILURE;
    } else {
	mobo_logf( LOG_INFO "SERIAL: transferred %d bytes from COM1->COM2\n",
			BYTES_XFER );
    }

    setstate( ISADEV_SERIAL, "COM2->COM1" );
    if ( test_xfer( DEV_COM2, DEV_COM1 ) != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "SERIAL: error in transfer from COM2 to COM1\n");
	setstate( ISADEV_SERIAL, "failed" );
	return STATUS_FAILURE;
    } else {
	mobo_logf( LOG_INFO "SERIAL: transferred %d bytes from COM2->COM1\n",
			BYTES_XFER );
    }

    mobo_logf( LOG_INFO "SERIAL: loopback test passed\n");
    setstate( ISADEV_SERIAL, str_passed );
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Main driver routine */

typedef DBM_STATUS (*tstfun)(void );

static tstfun devtst[ISADEV_NDEVS] = {
	cmos_test,
	keyb_test,
	mouse_test,
	floppy_test,
	flash_test,
	serial_test,
};

DBM_STATUS isadevs( int argc, char *argv[] )
{
    int i;
    DBM_STATUS r;
    DBM_STATUS result = STATUS_SUCCESS;
    String interp;

    diags_subsys_stat( SYS_ISA, DEV_PROBING );

    initstate();		/* draw user interface, setup data */

    for ( i = 0; i < ISADEV_NDEVS; i++ )
    {
	setstate( i, str_testing );
	r = devtst[i]();

        if ( r == STATUS_FAILURE )
	    result = STATUS_FAILURE;

	/* Only pass on warnings if nothing more drastic has already happened */
	if ( r == STATUS_WARNING ) 
	    if ( result != STATUS_FAILURE )
		result = r;
    }

    switch( result )
    {
	case STATUS_SUCCESS:
	    interp = "All tests PASSED";
	    diags_subsys_stat( SYS_ISA, DEV_PASSED );
	    break;

	case STATUS_WARNING:
	    interp = "All tests PASSED, but with WARNINGS incurred";
	    diags_subsys_stat( SYS_ISA, DEV_PASSED );
	    result = STATUS_SUCCESS; 		/* Actually we passed (ish) */
	    break;

	case STATUS_FAILURE:
	default:				/* (defensive) */
	    interp = "Errors were found, test FAILED";
	    diags_subsys_stat( SYS_ISA, DEV_FAILED );
	    break;
    }
    mobo_alertf( str_title, interp );
    mobo_zap( r_app );

    return result;
}

