/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* SMP-aware Memory test master driver applet                         */
/* (c) 1999 Alpha Processor Inc.                                      */
/* Begun by Stig Telfer, API, 4 August 1999			      */

#include "lib.h"
#include "uilib.h"
#include "memory.h"
#include "northbridge.h"
#include "platform.h"
#include "smp.h"
#include "osf.h"
#include "cmos_rtc.h"

#include "mem.h"			/* module's local header file */

/*--------------------------------------------------------------------*/

#define KEYPAUSE 100		/* SROM fix: delay for keypress in msec */

unsigned memtest_secs = 0;		/* back-door to set time limit */

static volatile unsigned iter;
static volatile unsigned errflag;
static SMP_ptr start, end;
static volatile size_t size;		/* test size in bytes */
static volatile unsigned ncpus;

extern int corrected_error_count;

static void setstate( void );

static String mem_title = "Memory Stress Test";		/* box title */
static String cpu_fmt = "CPU %d:      Write: %4dMB/s     Read: %4dMB/s   Size: %d MB";


/*--------------------------------------------------------------------*/
/* User interface handling code - the memory tests talk to the user via
 * these routines only */

/* Memory error-logging function - all IO from memtests comes through
   here, making a change in IO model easier to implement */

void mem_logerr( const String tst_name, const String tst_symptom,
                 SMP_ptr physaddr,
                 unsigned long actual, unsigned long expected )
{
        mobo_alertf("MEMORY ERROR", "%s test, %s; phys addr 0x%016lX\n"
                    "Read 0x%016lX, expected 0x%016lX, (diff=0x%016lX)",
                    tst_name, tst_symptom, physaddr, actual, expected,
                    actual^expected);
        errflag++;
}


static void usage( String name )
{
    mobo_alertf( "Usage", "Command line usage is:\r"
			  "%s <start-megs> <end-megs>", name );
}


/* All the data that is relevant to the user interface */

static struct {
        String test;
        char rd_wr;
        char sysstat[256];
	String cpustat[MAX_CPUS];
        unsigned emin, esec;
} disp = { "Setup", 'S', "Not yet probed for",
	   { NULL },
	   0, 0  };

static unsigned long sjiffies;

static void setstate( void )            /* update user interface */
{
    unsigned i;
    unsigned long curjif;

    /* timer update is put here, since it should be done on each call */
    curjif = jiffies - sjiffies;
    disp.esec = curjif / HZ;
    disp.emin = disp.esec / 60;
    disp.esec -= (disp.emin * 60);


    /* print out the static stuff */
    mobo_goto( p_app );
    printf_dbm( "Test:    %20s   Pass:     %7d%c  Elapsed: %4d:%02d\r"
		"Errors:     Corrected:  %5d   Uncorrected: %5d\r"
		"            System:     %5d   Processor:   %5d\r",
		disp.test, iter, disp.rd_wr, disp.emin, disp.esec,
		sys_crd + proc_crd, sys_mchk + proc_mchk,
		sys_crd + sys_mchk, proc_crd + proc_mchk );

    /* print a status message for each CPU present */
    /* The CPU's present are working according to logical ID so number 
     * contiguously */
    for ( i=0; i < smp_online_count; i++)
    {
	printf_dbm("%s\r", disp.cpustat[i] );
    }

    /* finish off with platform status */
    printf_dbm( disp.sysstat );	
}


void dump_cacheline( String name, SMP_ptr ptr, SMP_ptr exp)
{
    uint64 val0, val1, diff;
    unsigned i;
    Point P;
    int key;
    static const String quitmsg = "Press any key to continue or 'q' to abort";

    mobo_box( r_lrgapp, "Error!!!" );
    printf_dbm("CPU %d, %s test: Re-read of cacheline at 0x%016lx:\r",
                smp_phys_id(), name, (unsigned long)ptr & (~DBM_SUPER) );

    for( i = 0; i < 8; i++ )
    {
        val0 = *(uint64 *)exp++;
        val1 = *(uint64 *)ptr++;
        diff = val0 ^ val1;

        printf_dbm("wrote 0x%016lx, read 0x%016lx, diff 0x%016lx\r",
                    val0, val1, diff );
    }

    P.x = r_lrgapp.x + ( (r_lrgapp.w-strlen(quitmsg)) >> 1) ;
    P.y = r_lrgapp.y + r_lrgapp.h - 1;
    mobo_goto( P );
    printf_dbm( quitmsg );
    key = mobo_key( 0 );

    mobo_box( r_lrgapp, mem_title );		/* redraw box  */
    setstate();					/* and the stuff in it */
}



/*--------------------------------------------------------------------*/
/* memory test loop function */
/* On entry into this function, we are executing in SMP */

static DBM_STATUS memloop( int argc, char *argv[] )
{
    const tst_t *t;
    unsigned long stime, etime, ticks;
    size_t nbytes;
    unsigned mysize, imprime;
    unsigned readbw=0, writebw=0;
    char mycpustat[64];
    static volatile int qflag=0;
    static volatile int logging=0;
    unsigned myid = smp_myid();
    unsigned phys_id = smp_phys_id();

    /* Wait here because primary must set up some shared data first */
    sprintf_dbm( mycpustat, "CPU %d: Setting up...", phys_id );
#ifdef CONFIG_TINOSA
    mobo_logf( mycpustat );		/* DEBUG */
#endif
    smp_sync();

    mysize = (size / ncpus) >> 20;
    imprime = smp_primary();
    if ( imprime )
    {
	qflag = 0;
	logging = 0;
    }

    disp.cpustat[myid] = mycpustat;		/* set up my own CPU state */

    /*--------------------------------------------------------------------*/
    /* Memory test loop */

    for (iter = 0, errflag = 0, t = tst; ; t = tst + iter % ntests )
    {
	if ( imprime ) {
	    /* setup of display state for a new iteration */
	    if ( iter % ntests == 0 )
		plat_sysstat( disp.sysstat );	/* update on system status */
	    disp.test = t->name;
	    disp.rd_wr = 'W';
	}

	sprintf_dbm( mycpustat, cpu_fmt, phys_id, writebw, readbw, mysize );
	smp_sync();
	if ( imprime )	
	    setstate();


	/* Write out of data */

	smp_sync();		/* line up again to enter test together */
	stime = rpcc();					/* timing info */
	nbytes = t->write( start, size );
	etime = rpcc();					/* timing info */
	if ( stime >= etime ) etime += 1UL << 32;       /* wraparound? */
	ticks = etime - stime;           /* elapsed time in CPU cycles */
	nbytes = (nbytes + (1UL<<19)) >> 20;		/* round into mbytes */
	writebw = (nbytes * 1000000000000UL)/( primary_impure->CYCLE_CNT * ticks );

        sprintf_dbm( mycpustat, cpu_fmt, phys_id, writebw, readbw, mysize );
	smp_sync();			/* All block here for test results */

	if ( imprime ) 
	{
	    disp.rd_wr = 'R';
	    setstate();
	}



	/* Read back and verify data */
	smp_sync();		/* line up again to enter test together */
	stime = rpcc();                  /* timing info */
	nbytes = t->read( start, size );
	etime = rpcc();                  /* timing info */
	if ( stime >= etime ) etime += 1UL << 32;        /* wraparound? */
        ticks = etime - stime;           /* elapsed time in CPU cycles */
	nbytes = (nbytes + (1UL<<19)) >> 20;		/* round into mbytes */
        readbw = (nbytes * 1000000000000UL)/( primary_impure->CYCLE_CNT * ticks );


        sprintf_dbm( mycpustat, cpu_fmt, phys_id, writebw, readbw, mysize );
	smp_sync();			/* all block here for test results */

	/* End of iteration - does the user want to break out? */
	if ( imprime )
	{
	    if ( iter % 10 == 0 )		setstate();

	    if ( mobo_key( KEYPAUSE ) != -1 )		qflag += 1;

	    /* In an automated test the diags may limit the test duration */
	    if ( memtest_secs > 0 )
		if ( disp.emin * 60 + disp.esec > memtest_secs )
			qflag += 1;

	    iter++;			/* only primary increments the loop */
	}

	if ( logging )
        {
	    mobo_logf( LOG_INFO
		"MEM: pass %d CPU %d test %s (%d Mbytes) wrote %dMB/s read %dMB/s\n",
		iter, phys_id, disp.test, mysize, writebw, readbw );
        }

	smp_sync();
	if ( qflag != 0 )	break;
    }

    return STATUS_SUCCESS;
}



/*--------------------------------------------------------------------*/
/* Setup for SMP memory test (even on single-cpu) */

DBM_STATUS mem( int argc, char *argv[] )
{
    Point P;
    const String keymsg = "Press any key to quit";
    int base_pfn, n_pfns, pfn_alloc;

    /* look for command line arguments for the range */
    if ( argc == 1 )
    {
	/* Use all memory available */
	page_max_free( &base_pfn, &n_pfns );

	/* Note: we want each CPU to have a round number of pages */
	pfn_alloc = 8 * smp_online_count;	/* 8-page test granularity */
	n_pfns -= n_pfns % pfn_alloc;

	start = (SMP_ptr)PFN2ADDR( base_pfn, DBM_SUPER );
	end   = (SMP_ptr)PFN2ADDR( base_pfn + n_pfns, DBM_SUPER );
    }
    else if ( argc == 3 )		/* specific test range */
    {
	/* FIXME: this could be dangerous, as the user could specify a range
	 * that trashes all of diags.  No warning is given for dodgy ranges */
	start = (SMP_ptr)( DBM_SUPER | (unsigned long)atoi( argv[1] ) << 20 );
	end   = (SMP_ptr)( DBM_SUPER | (unsigned long)atoi( argv[2] ) << 20 );

	base_pfn = ADDR2PFN( start, DBM_SUPER );
	n_pfns = ADDR2PFN( end, DBM_SUPER ) - base_pfn;

    } else {
	usage( argv[0] );
	return STATUS_FAILURE;
    }

    if ( end <= start )			/* arg validation */
    {
	usage( argv[0] );
	return STATUS_FAILURE;
    }


    size = (end - start) * sizeof( start[0] );		/* in bytes */

    /* Mark our claim to the memory manager */
    page_mark_range( base_pfn, n_pfns, TEMPORARY_PAGE );

    /* system setup */
    plat_intclr();
    plat_nmiclr();


    /* display setup */
    diags_subsys_stat( SYS_MEM, DEV_STRESSING );
    /* Note: if we use long parameters in the message below, we get KSEG bits,
     * which are confusing.  Beware tests reaching beyond 4GB...
     */
    mobo_logf( "MEM: %ldMB memory test over range 0x%X->0x%X\n",
		size >> 20, start, end );
    mobo_box( r_lrgapp, mem_title );

    /* gets printed even in batch mode, in case a user is watching... */
    P.x = r_lrgapp.x + ( (r_lrgapp.w-strlen(keymsg)) >> 1) ;
    P.y = r_lrgapp.y + r_lrgapp.h - 1;
    mobo_goto( P );
    printf_dbm( keymsg );

    sjiffies = jiffies;			/* start the clock */
    ncpus = smp_ncpus();


    /*--------------------------------------------------------------------*/
    /* Launch SMP code... */

    smp_exec( memloop );

    /*--------------------------------------------------------------------*/
    /* Cleanup and log results */

    /* Rescind our claim on the memory block */
    page_mark_range( base_pfn, n_pfns, FREE_PAGE );

    memtest_secs = 0;		/* implicitly disables test duration limit */
    mobo_logf( LOG_INFO "MEMORY: test over %d Mbytes completed after %d:%02d\n",
		size >> 20, disp.emin, disp.esec );

    mobo_zap( r_lrgapp );

    if ( proc_mchk == 0 && proc_crd == 0 && sys_mchk == 0 && sys_crd == 0 )
    {
        diags_subsys_stat( SYS_MEM, DEV_PASSED );
	return STATUS_SUCCESS;
    } else {
	diags_subsys_stat( SYS_MEM, DEV_FAILED );
	return STATUS_FAILURE;
    }
}

