/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* I2C System management high-level access routines */

#undef TRACE_ENABLE

#include <stdarg.h>

#include "lib.h"
#include "uilib.h"
#include "specifics.h"
#include "platform.h"

#include "i2c.h"
#include "i2c/lm75.h"
#include "i2c/adm9240.h"
#include "i2c/pcf8574.h"
#include "i2c/pcf-eprom.h"


static DBM_STATUS do_thermal( const I2Cbus_t *I, String buf )
{
    DBM_STATUS sval = STATUS_FAILURE;
    i2c_therm_t T;

    switch( I->chip )
    {
	case I2CCHIP_LM75:
	    sval = lm75_read( I->addr, &T );
	    break;

	case I2CCHIP_ADM9240:
	    sval = adm_therm_read( I->addr, &T );
	    break;

	default:
	    sval = STATUS_FAILURE;
	    break;
    }

    if ( sval != STATUS_FAILURE )
    {
	sprintf_dbm( buf, "Temperature=%d.%d C, Max=%d.%d C, Hyst=%d.%d C",
		T.temp / 10, T.temp % 10, T.max / 10, T.temp % 10,
		T.hyst / 10, T.hyst % 10 );

	/* Are we overheating on this sensor? */
	if ( T.temp > T.max )
	    sval = STATUS_WARNING;

    } else {
	sprintf_dbm( buf, "Unrecognised chip or failed read" );
    }

    return sval;
}


static DBM_STATUS do_fan( const I2Cbus_t *I, String buf )
{
    DBM_STATUS sval = STATUS_FAILURE;
    i2c_fan_t F;

    switch( I->chip )
    {
	case I2CCHIP_ADM9240:
	    sval = adm_fan_read( I->addr, I->sensor, &F );
	    break;

	default:
	    sval = STATUS_FAILURE;
	    break;
    }

    if ( sval != STATUS_FAILURE )
    {
	sprintf_dbm( buf, "Fan speed = %d RPM, Minimum = %d RPM",
		F.rpm, F.threshold );

	/* Is our fan RPM too low? */
	if ( F.rpm < F.threshold )
	    sval = STATUS_WARNING;

    } else {
	sprintf_dbm( buf, "Unrecognised chip or failed read" );
    }

    return sval;
}


static DBM_STATUS do_volt( const I2Cbus_t *I, String buf )
{
    DBM_STATUS sval = STATUS_FAILURE;
    i2c_volt_t V;

    V.nominal = 0;			/* need a way of setting this */

    switch( I->chip )
    {
	case I2CCHIP_ADM9240:
	    sval = adm_voltages( I->addr, I->sensor, &V );
	    break;

	default:
	    sval = STATUS_FAILURE;
	    break;
    }

    if ( sval != STATUS_FAILURE )
    {
	sprintf_dbm( buf, "Measured at %d.%dV, Max = %d.%dV, Min = %d.%dV",
		V.v / 10, V.v % 10,
		V.max / 10, V.max % 10,
		V.min / 10, V.min % 10 );

	/* Is our voltage out of range? */
	if ( V.v > V.max || V.v < V.min )
	    sval = STATUS_WARNING;

    } else {
	sprintf_dbm( buf, "Unrecognised chip or failed read" );
    }

    return sval;
}


static DBM_STATUS do_regmux( const I2Cbus_t *I, String buf )
{
    uint8 muxval;
    DBM_STATUS sval;

    switch( I->chip )
    {
	case I2CCHIP_PCF8574:
	    sval = pcf8574_get( I->addr, &muxval );
	    break;

	default:
	    sval = STATUS_FAILURE;
	    break;
    }

    if ( sval == STATUS_SUCCESS )
	sprintf_dbm( buf, "Set to 0x%02X", muxval );
    else
	sprintf_dbm( buf, "Unrecognised chip or failed read" );

    return sval;
}


/* The number of bytes to read from an EPROM to get a gist of content */
#define BYTES_READ 32

static DBM_STATUS do_eprom( const I2Cbus_t *I, String buf )
{
    char data[ BYTES_READ ];
    DBM_STATUS sval;
    int i;

    switch( I->chip )
    {
	case I2CCHIP_PCF8582:				/* 256-byte EEPROM */
	    sval = pcfeprom_read( I->addr, 0, BYTES_READ, data );
	    break;

	default:
	    sval = STATUS_FAILURE;
	    break;
    }


    if ( sval == STATUS_SUCCESS )
    {
	/* Convert the data in-situ into readable ASCII */
	for( i=0; i < BYTES_READ - 1; i++ )
	{
	    data[i] = isprint( data[i] ) ? data[i] : '.';
	}
	data[ BYTES_READ - 1 ] = '\0';		/* NULL-termination */

	sprintf_dbm( buf, "Begins '%s...'", data );
    }
    else
    {
	sprintf_dbm( buf, "Unrecognised chip or failed read" );
    }

    return sval;
}


/*--------------------------------------------------------------------*/
/* Print device-dependent status (eg temp, fan RPM) to the buffer supplied */

DBM_STATUS i2c_status( const I2Cbus_t *I, String buf )
{
    DBM_STATUS sval = STATUS_FAILURE;
    DBM_STATUS select;

    /* Has the user specified a device that exists on our I2C bus? */
    if ( I == NULL )
        return STATUS_FAILURE;

    TRACE( "Examining %d %d %d (%s)\n", I->dev, I->sensor, I->chip, I->desc );

    select = plat_i2c_select( I );
    switch( select ) 
    {
	/* If a device is not plugged in it should not be an error */
        case STATUS_WARNING:
            sprintf_dbm( buf, "Device not available in configuration" );
	    return STATUS_SUCCESS;
        
	/* If during configuration of a device there is an error, its bad */
        case STATUS_FAILURE:
            sprintf_dbm( buf, "Severe error in accessing device!" );
	    return STATUS_FAILURE;

	/* Otherwise, everything went to plan... */
        default:
            break;
    }

    switch( I->sensor )
    {   
        case I2CSENS_TEMP0:
	case I2CSENS_TEMP1:
            sval = do_thermal( I, buf );
            break;

	case I2CSENS_FAN0:
	case I2CSENS_FAN1:
	    sval = do_fan( I, buf );
	    break;
        
	case I2CSENS_REGMUX:
	    sval = do_regmux( I, buf );
	    break;

	case I2CSENS_EPROM:
	    sval = do_eprom( I, buf );
	    break;

	case I2CSENS_2_5V:
	case I2CSENS_3_3V:
	case I2CSENS_5V:
	case I2CSENS_12V:
	case I2CSENS_VCCP1:
	case I2CSENS_VCCP2:
	    sval = do_volt( I, buf );
	    break;

        default:
	    sprintf_dbm( buf, "Not implemented yet!" );
	    sval = STATUS_SUCCESS;		/* don't let it spoil the fun */
            break;
    }
   
    return sval;
}


/*--------------------------------------------------------------------*/
/* I2C POST (power-on self-test) routine */

#define MAXREPORTLEN 80

typedef struct _i2c_report {
	char info[ MAXREPORTLEN ];
	struct _i2c_report *next;
} i2c_report;

static i2c_report *i2c_failure_list = NULL;


static void add_to_failures( const String fmt, ... )
{
    va_list ap;
    i2c_report *R;

    va_start( ap, fmt );

    R = malloc( sizeof( i2c_report ) );
    if ( R==NULL )			/* can't do anything right! */
	return;

    vsprintf_dbm( R->info, fmt, ap );
    R->next = i2c_failure_list;
    i2c_failure_list = R;

    va_end( ap );
}


/* If i2c_post returned anything other than STATUS_SUCCESS, a console should 
 * be brought up by the POST routine and this function then called to dump info
 * about what is amiss.
 *
 * This function also has the side-effect of freeing the malloc'ed data.
 */

void i2c_post_report( void ) 
{
    i2c_report *tmp;
    int max_screen_reports;
    int err_count=0;
    Point P;

    max_screen_reports = r_lrgapp.h - 2;

    mobo_logf( LOG_CRIT "I2C: System health alert!\n" );
    mobo_box( r_lrgapp, "System health alert!" );
    mobo_goto( p_app );

    while( i2c_failure_list != NULL )
    {
	mobo_logf( LOG_CRIT "%s\n", i2c_failure_list->info );
	if ( err_count < max_screen_reports )
	    printf_dbm( "%s\r", i2c_failure_list->info );
	else if ( err_count == max_screen_reports )
	    printf_dbm( "[more errors follow and are not shown here]" );


	/* free our data structures */
	tmp = i2c_failure_list;
	i2c_failure_list = i2c_failure_list->next;
	free( tmp );
	err_count++;
    }

    /* Add a 'press a key' message */
    P.x = r_lrgapp.x + ( (r_lrgapp.w-strlen( s_ekey )) >> 1 );
    P.y = r_lrgapp.y + r_lrgapp.h - 1;
    mobo_goto( P );
    printf_dbm( s_ekey );
    mobo_key( 0 );

    /* At this point there have been system health monitor failures or alerts. 
     * We may wish to shut down the machine in this event to let things cool off
     * Currently however we're choosing to alert and then do nothing...
     */

    mobo_zap( r_lrgapp );		/* Clean up and quit */
}


#define FAN_TIMEOUT	8		/* in seconds */

DBM_STATUS i2c_post( void )
{
    DBM_STATUS sval, final_sval=STATUS_SUCCESS;
    const I2Cbus_t *I;
    char buf[80];
    int i;

    /* Initialise the I2C bus and configure all sensors */
    sval = plat_i2cinit();
    if ( sval == STATUS_FAILURE )
    {
	add_to_failures( "I2C failure in general setup and configuration" );
	return sval;
    }

    /* Often gets left until last on partially-implemented platforms :-) */
    if ( plat_i2c_bus == NULL )
        return STATUS_SUCCESS;		/* handle silently */

    for( I=plat_i2c_bus; I->dev != I2C_NODEV; I++ )
    {
	sval = i2c_status( I, buf );

	/* DEBUG: on power-up the fans may not be spinning at full speed	
	 * by the time this POST routine is run.  Try a little harder to get
	 * a good reading from the fans... */
	if ( I->sensor == I2CSENS_FAN0 || I->sensor == I2CSENS_FAN1 )
	{
	    for( i=0; i<FAN_TIMEOUT; i++ )
	    {
		if ( sval == STATUS_SUCCESS )
		    break;

		mobo_logf( LOG_WARN
		    "I2C: self-test found fan not at full speed, "
		    "may be spinning up...\n"
		    "%s: %s\n", I->desc, buf );

		sleep( 1 );
		sval = i2c_status( I, buf );		/* try another read */
	    }
	}

	if ( sval == STATUS_FAILURE )
	{
	    add_to_failures( "Failure on read of '%s' at I2C address 0x%02X",
		I->desc, I->addr );

	    final_sval = STATUS_FAILURE;
	}

	/* A warning from the I2C suggests that something's beyond spec */
	if ( sval == STATUS_WARNING )
	{
	    /* Ahem, here we use some inside knowledge that the failures are
	     * listed in reverse order, so we log second line first ... */

	    add_to_failures( "  Reads: %s", buf );
	    add_to_failures( "Sensor '%s' is reading out of range:", I->desc );

	    /* don't overwrite any previous more drastic event */
	    if ( final_sval != STATUS_FAILURE )
		final_sval = STATUS_WARNING;
	}
    }

    return final_sval;
}



/*--------------------------------------------------------------------*/
/* Lookup a particular device (if available) on the platform I2C bus */

const I2Cbus_t *i2c_lookup( const i2c_device D )
{
    const I2Cbus_t *I = plat_i2c_bus;

    for( I = plat_i2c_bus; I->dev != I2C_NODEV; I++ )
	if ( D==I->dev )
	 	return I;

    /* Device not found? */
    return NULL;
}

