#ifndef __ROMHEAD_H_LOADED
#define __ROMHEAD_H_LOADED
/*****************************************************************************

       Copyright  1993, 1994 Digital Equipment Corporation,
                       Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/

/*
 *
 *	Special ROM header
 *	==================
 *	The System ROM can contain multiple ROM images, each with
 *	its own header. That header tells the SROM where to load
 *	the image and also if it has been compressed with the
 *	"makerom" tool.  For System ROMs which contain a single
 *	image, the header is optional.  If the header does not
 *	exist the complete System ROM is loaded and executed at
 *	physical address zero.
 *                                                         +-- Offset
 *                                                         |         Header
 *	31                                             0   |     +-- Revisions
 *	+-----------------------------------------------+  |     |   Supported
 *	|   VALIDATION PATTERN 0x5A5AC3C3               | 0x00  all
 *	+-----------------------------------------------+
 *	|   INVERSE VALIDATION PATTERN 0xA5A53C3C       | 0x04  all
 *	+-----------------------------------------------+
 *	|   HEADER SIZE (Bytes)                         | 0x08  all
 *	+-----------------------------------------------+
 *	|   IMAGE CHECKSUM                              | 0x0C  all
 *	+-----------------------------------------------+
 *	|   IMAGE SIZE (Memory Footprint)               | 0x10  all
 *	+-----------------------------------------------+
 *	|   DECOMPRESSION FLAG                          | 0x14  all
 *	+-----------------------------------------------+
 *	|   DESTINATION ADDRESS LOWER LONGWORD          | 0x18  all
 *	+-----------------------------------------------+
 *	|   DESTINATION ADDRESS UPPER LONGWORD          | 0x1C  all
 *	+-----------------------------------------------+
 *     	|   FIRMWARE ID <15:8> | HEADER REV <7:0>       | 0x20  1+
 *	|   Reserved <31:24>   | HEADER REV EXT <23:16> |
 *	+-----------------------------------------------+
 *	|   ROM IMAGE SIZE                              | 0x24  1+
 *	+-----------------------------------------------+
 *	|   OPTIONAL FIRMWARE ID <31:0>                 | 0x28  1+
 *	+-----------------------------------------------+
 *	|   OPTIONAL FIRMWARE ID <63:32>                | 0x2C  1+
 *	+-----------------------------------------------+
 *	|   ROM OFFSET<31:2>     | ROM OFFSET VALID<0>  | 0x30  2+
 *	+-----------------------------------------------+
 *	|   HEADER CHECKSUM (excluding this field)      | 0x34  1+
 *	+-----------------------------------------------+
 *
 *	VALIDATION PATTERN
 *	------------------
 *	The first quadword contains a special signature pattern
 *	that is used to verify that this "special" ROM header
 *	has been located.  The pattern is 0x5A5AC3C3A5A53C3C.
 *
 *	HEADER SIZE (Bytes)
 *	-------------------
 *	The header size is the next longword.  This is provided
 *	to allow for some backward compatibility in the event that
 *	the header is extended in the future.  When the header
 *	is located, current versions of SROM code determine where
 *	the image begins based on the header size.  Additional data
 *	added to the header in the future will simply be ignored
 *	by current SROM code. Additionally, the header size = 0x20
 *	implies version 0 of this header spec.  For any other size
 *      see HEADER REVISION to determine header version.
 *      
 *      
 *	IMAGE CHECKSUM
 *	--------------
 *	The next longword contains the image checksum.  This is
 *	used to verify the integrity of the ROM.  Checksum is computed
 *      in the same fashion as the header checksum.
 *      Although this field was provided with version 0 of this header
 *      spec, the checksum was not really computed until version 1.
 *
 *	IMAGE SIZE (Memory Footprint)
 *	-----------------------------
 *	The image size reflects the size of the image after it has
 *	been loaded into memory from the ROM. See ROM IMAGE SIZE.
 *
 *	DECOMPRESSION FLAG
 *	------------------
 *	The decompression flag tells the SROM code if the makerom
 *	tool was used to compress the ROM image with a "trivial
 *	repeating byte algorithm".  The SROM code contains routines
 *	which perform this decompression algorithm.  Other
 *	compression/decompression schemes may be employed which work
 *	independently from this one.
 *
 *	DESTINATION ADDRESS
 *	-------------------
 *	This quadword contains the destination address for the
 *	image.  The SROM code  will begin loading the image at this
 *	address and subsequently begin its execution there.
 *
 *	HEADER REV
 *	----------
 *	The revision of the header specifications used in this
 *	header.  This is necessary to provide compatibility to
 *	future changes to this header spec.  Version 0 headers
 *	are identified by the size of the header. See HEADER SIZE.
 *	For Version 1 or greater headers this field must be set to
 *	a value of 1.  The header revision for version 1 or greater
 *	headers is determined by the sum of this field and the
 *	HEADER REV EXT field. See HEADER REV EXT.  
 *
 *	FIRMWARE ID
 *	-----------
 *	The firmware ID is a byte that specifies the firmware type.
 *	This facilitates image boot options necessary to boot
 *	different operating systems.
 *
 *		  firmware
 *	firmware    type
 *	--------  --------
 *	  DBM	     0	     Alpha Evaluation Boards Debug Monitor
 *	  WNT        1       Windows NT Firmware
 *	  SRM        2       Alpha System Reference Manual Console
 *	  FSB	     6	     Alpha Evaluation Boards Fail-Safe Booter
 *	  Milo       7       Linux Miniloader
 *	  SROM      10       Serial ROM (SROM) Image 
 *
 *	HEADER REV EXT
 *	--------------
 *	The header revision for version 1 or greater headers is
 *	determined by the sum of this field and the HEADER REV
 *	field. See HEADER REV.  
 *
 *	ROM IMAGE SIZE
 *	--------------
 *	The ROM image size reflects the size of the image as it is
 *	contained in the ROM. See IMAGE SIZE.
 *
 *	OPTIONAL FW ID
 *	--------------
 *	This is an optional field that can be used to provide
 *	additional firmware information such as firmware revision
 *	or a character descriptive string up to 8 characters.
 *
 *	ROM OFFSET
 *	----------
 *	This field specifies the default ROM offset to be used
 *	when programming the Image into the ROM.
 *
 *	ROM OFFSET VALID
 *	----------------
 *	The lower bit of the ROM OFFSET field should be set when
 *	the ROM OFFSET field is specified.  When no ROM OFFSET is
 *	specified the ROM OFFSET and VALID fields will contain zero.
 *
 *	HEADER CHECKSUM
 *	---------------
 *	The checksum of the header.  This is used to validate
 *	the presence of a header beyond the validation provided
 *	by the validation pattern.  See VALIDATION PATTERN.
 *	The header checksum is computed from the beginning of
 *	the header up to but excluding the header checksum
 *	field itself.  If there are future versions of this
 *	header the header checksum should always be the last
 *	field defined in the header.  The checksum algorithm used
 *	is compatible with the standard BSD4.3 algorithm provided
 *	on most implementations of Unix.  Algorithm: The checksum
 *	is rotated right by one bit around a 16 bit field before
 *	adding in the value of each byte.
 *
 */

/* API Addition: */
/* This interface distances headers from the body of a ROM image, so that
 * they no longer need to be joined together.  This creates scope for the 
 * ROM to have a "table of contents" concept, speeding access to images
 * further up in the ROM.  The danger: unless a table of contents contains
 * redundant headers, if it got wiped we'd *never* know what is what...
 */


/*------------------------------------------------------------------------*/
/* ROM header definitions */

#define ROM_H_SIGNATURE 0x5A5AC3C3U
#define ROM_H_REVISION  2
#define DEFAULT_ROM_H_REVISION 2
#define ROM_H_BRANCH_SIGNATURE 0xc3e00000
#define ROM_H_BRANCH_OFFSET_MASK 0x1fffff

/* Structure used by Compaq Firmware */
typedef struct {
    unsigned char build_minute;
    unsigned char build_hour;
    unsigned char build_day;
    unsigned char build_month;
    unsigned char build_year;
    unsigned char revision;
    unsigned char minor_version;
    unsigned char major_version;
} cpq_fwid_t;

/* Structure used by API Firmware */
typedef struct {
    uint8 major;	/* Firmware major version */
    uint8 minor;	/* Firmware minor version */
    uint8 rev;		/* Firmware miniscule version */
    uint8 api_category;	/* API part number category */
    uint8 api_major;	/* API part number major byte */
    uint8 api_minor;	/* API part number minor byte */
    uint8 api_rev;	/* API part revision number (0-9) */
    uint8 api_letter;	/* letter (A-Z starting from 0=A) */
} api_fwid_t;

typedef union {
    struct {
	/*
	 * Version 0 definition of the ROM header.
	 */
	struct {
	    ui signature;	/* validation signature                    */
	    ui csignature;	/* inverse validation signature            */
	    ui hsize;		/* header size                             */
	    ui checksum;	/* checksum                                */
	    ui size;		/* image size (Memory Footprint)           */
	    ui decomp;		/* decompression algorithm                 */
	    struct {
		ui low;
		ui high;
	    } destination;	/* destination address                     */
	} V0;

	/*
	 * Version 1 extensions to the ROM header.
	 */
	struct {
	    char hversion;	/* ROM header version   (Byte 0)            */
	    char fw_id;		/* Firmware ID          (Byte 1)            */
	    char hversion_ext;	/* header version ext   (Byte 2)            */
	    char reserved;	/* Reserved             (Byte 3)            */
	    ui rimage_size;	/* ROM image size                           */

	    /* The firmware optional ID is a bit of a mess due to a mishmash
	     * of conflicting formats with no clear indication which one a
	     * particular firmware header is following.
	     * The various options are presented here.
	     */
	    union {

		/* Optional Firmware ID (character array)   */
		char id[8];

		/* Structure used by Compaq Firmware */
		cpq_fwid_t fw_id_S;

		/* Structure used by API Firmware */
		api_fwid_t api_id_S;

	    } fwoptid;
	} V1;

	/*
	 * Version 2 extensions to the ROM header.
	 */
	struct {
	    ui rom_offset;	/* ROM Offset<31:2>, ROM Offset Valid<0> */
	} V2;

	/* 
	 * Version 3 (API NetWorks unofficial extension)
	 * Features the ability to have the header disjoint from, or appended to
	 * the main ROM image.  This is required for SROM/CBOX images that must
	 * begin with image data, due to EPLD shortcomings.
	 * Version 3 adds no fields but enforces the use of V2 rom_offset as the
	 * start of the body of the ROM.
	 * Implementation note: this means that the header (and header checksum)
	 * will change depending on where you put the image in the ROM.
	 */

	/*
	 * Future extensions to the header should be included before
	 * this header checksum.  (See HEADER CHECKSUM description)
	 */
	ui hchecksum;		/* Header checksum, (Always last entry)     */
    } romh;
    ui romh_array[1];		/* To allow longword access to the data     */
} romheader_t;

/*
 * Registered Firmware types.
 */
typedef enum _fwids {	FW_DBM=0,		/* Debug Monitor */
			FW_WNT=1,		/* AlphaBIOS / ARC Firmware */
			FW_SRM=2,		/* SRM Console */
			FW_FSB=6,		/* Fail-safe boot firmware */
			FW_MILO=7,		/* Milo (Linux Miniloader) */
			FW_VXW=8,		/* VxWorks */
			FW_DIAG=9,		/* Diagnostics */
			FW_SR=10,		/* SROM Code */
			FW_LINUX=11,		/* Linux kernel */
			FW_CBOX=12,		/* CBOX config table */
			FW_OSFPAL=13,		/* OSF PALcode */
			FW_NVRAM=15,		/* NVRAM data area */
			FW_NVLOG=16,		/* NVRAM diagnostic log */
			FW_NVENV=17,		/* NVRAM diagnostic env vars */
			FW_INITRD=20,		/* Linux InitRD image */
			FW_UNKNOWN=-1,
			FW_EMPTY=-2		/* Empty region of ROM */
} FWID;


/*
 * The ROM header checksum should always be assigned to the last
 * field in the header.  Therefore, when reading headers created
 * by various versions of makerom the ROM header checksum can be
 * in different locations.  This macro can be used to access the
 * ROM header checksum in the proper location.
 */
#define ROM_HEADER_CHECKSUM(x) \
  ((x)->romh_array[((x)->romh.V0.hsize - sizeof((x)->romh.hchecksum))/sizeof(ui)])

/*
 * Macro to provide the header version number
 */
#define ROMH_VERSION(x) ((x)->romh.V0.hsize == 0x20 ? 0 : \
			 ((x)->romh.V1.hversion+(x)->romh.V1.hversion_ext))

/*
 * Macro to assist in computing the BSD4.3 style checksum.
 */
#define ROTATE_RIGHT(x) if ((x) & 1) (x) = ((x) >>1) + 0x8000; else (x) = (x) >>1;

/*
 * Macro used to increment the checksum
 * by a new byte while keeping the total
 * checksum within the 16 bit range.
 */
#define COMPUTE_CHECKSUM(c,k) \
   {ROTATE_RIGHT(k); k += (ub) c; k &= 0xffff;}


/*
 * Functions for manipulating memory-resident ROM headers
 */

/* From header data (if available) deduce the FW ID of the ROM image */
extern FWID rom_fwid( const romheader_t *H );

/* From header data and location in ROM, return ROM address of body */
extern unsigned rom_body_base( const romheader_t *H, const unsigned romptr );

/* From header data, deduce the size of the ROM image */
extern unsigned rom_body_size( const romheader_t *H );

/* Is a header (memory resident copy) a valid, well-formed ROM header? */
extern BOOLEAN rom_hdr_valid( const romheader_t *H );

/* A function to parse a header into one of its multitudinous interpretations */
extern String rom_parseRevision( const romheader_t *H, String fwidstr );

/* Logs details about the ROM header supplied */
extern void rom_dump_hdr( const romheader_t * );



/*------------------------------------------------------------------------*/
/* ROM Body definitions */

typedef unsigned char rombody_t;

/*
 * Functions for manipulating memory-resident ROM bodies
 */

/* Is an image (memory-resident copy) a valid, well-formed ROM image? */
extern BOOLEAN rom_body_valid( const romheader_t *, const rombody_t * );



/*------------------------------------------------------------------------*/
/* System data structure for managing the ROM layout */

typedef struct _hdr {
	unsigned hstart;		/* header start in ROM */
        unsigned rstart, rsize;         /* body start and size in ROM */
        unsigned astart, asize;         /* available space for body use/reuse */

        unsigned char flags;		/* see flags variables below */
	char fwid;			/* firmware ID */

	String name;			/* brief name describing this image */
	String defcol;			/* default colouring for graphic map */ 

        romheader_t *hdr;		/* header data, as described above */
	rombody_t *body;		/* body data of ROM image */

        struct _hdr *next;
} rom_t;

/* Settings for flags in the above */
#define ROM_IMGVALID	0x01		/* Image checksum is valid */
#define ROM_IMGCSUM	0x02		/* Represents a checksummable object */
#define ROM_PLATRSVD	0x04		/* platform defines this region */
#define ROM_EMPTYRGN	0x08		/* header represents empty region */
#define ROM_RAMIMG	0x10		/* Image is present in BootMem */


/* Linked list built by header scan incorporating platform reserved regions */
extern rom_t *rom_phys_map;

/* Linked list built by ROM-in-RAM (BootMem) downloaded images */
extern rom_t *rom_ram_map;

#if 0	/* Not currently used FIXME ROM BODGE */
/* Linked list of ROM images of all kinds and from all sources */
extern rom_t *rom_map;
#endif

/* Has the ROM been fully mapped and indexed yet? */
extern BOOLEAN rom_mapped;


/*
 * Functions for handling ROM images contained in one of the lists above
 */

/* This function is a good place to start with any ROM activity */
/* It builds a ROM map, in plat_phys_rom, with platform reserved regions
 * and empty space also tracked.
 */
extern DBM_STATUS rom_index( void );

/* Index the physical ROM for images, which regenerates rom_phys_map */
extern DBM_STATUS rom_phys_scan( void );

/* Index the physical ROM for images, which regenerates rom_phys_map */
extern DBM_STATUS rom_rsvd_scan( void );

/* Index the BootMem memory region for images, which regenerates rom_ram_map */
extern DBM_STATUS rom_ram_scan( unsigned char *, const size_t );

/* Clear a rom_t structure, created by one of the above routines */
/* NB: if you free a global rom_t structure, it must be set to NULL after! */
extern DBM_STATUS rom_free( rom_t *R );


/* Find an image from a given list by FW ID */
extern rom_t *rom_search( const FWID I, rom_t *list );

/* Given descriptor, load body data into memory */
/* [set dest non-NULL to override destination address for ROM load] */
extern char *rom_load( rom_t *, char * );

/* Given descriptor, load and execute firmware image */
extern DBM_STATUS rom_exec( rom_t * );

/* Logs details about the ROM image supplied */
extern void rom_dump( const rom_t * );


/*--------------------------------------------------------------------*/
/* General widgets for manipulating ROM images */

/* Returns short name of FWID I */
extern String rom_I2name( FWID I );

/* Returns ANSI escape sequence of designated colour of FWID I */
extern String rom_I2colour( FWID I );





/* Checksumming widgetry */
extern unsigned short compute_checksuml( unsigned val, unsigned short total );
extern unsigned short compute_checksumb( unsigned char v, unsigned short t );





/* Load an image from some other location in RAM and jump to it. */
extern void rom_RAMload( romheader_t * );


/* A quick check to see if a header is in one of the more likely places 
 * (ie a sector boundary).  Run this first when seeking a particular image
 * and it will save a few seconds load time */
extern rom_t *rom_quickscan( FWID I );



#endif /* __ROMHEAD_H_LOADED */
