/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Goby / Sturgeon / Caspian chipset support.  This chipset, developed 
 * by API, is derivative of the AMD Irongate chipset and has a similar 
 * interface.  It also features the Lightning Data Transport IO-bus
 * for high speed connection to a wide range of IO buses and devices
 */

#define TRACE_ENABLE		/* define to enable debugging output */
#define DEBUG_GOBY_64BIT	/* Goby p1 has issues with 64-bit CSR writes? */
#define DEBUG_IO( format, args... )

#include "lib.h"
#include "uilib.h"
#include "pci.h"
#include "memory.h"

#include "specifics.h"
#include "northbridge.h"

/*--------------------------------------------------------------------*/
/* Machine configuration data */

/* number of independent PCI bus trees */
const unsigned char nb_pci_hosecount=1;


/*--------------------------------------------------------------------*/
/* Low level IO space access mechanisms */

#define SHIFTVAL 24
#define PCI_MEM		( DBM_SUPER | (0x80000UL << SHIFTVAL))
#define PCI_IACK	( DBM_SUPER | (0x801F8UL << SHIFTVAL))
#define PCI_IO		( DBM_SUPER | (0x801FCUL << SHIFTVAL))
#define PCI_CSR		( DBM_SUPER | (0x801FEUL << SHIFTVAL))

static uint8 *caspian_iospace = (uint8 *)PCI_IO;

typedef uint8 *puint8;

void outportb(ui p , ui d)
{
    uint8 *ioport = caspian_iospace + p;
    DEBUG_IO("0x%X -> 0x%X\n", p, d );
    *ioport = d & 0xFFU;
    mb();
}

void outportw(ui p , ui d)
{
    uint16 *ioport = (uint16 *)(caspian_iospace + p);
    *ioport = d & 0xFFFFU;
    DEBUG_IO("0x%X -> 0x%X\n", p, d );
    mb();
}

void outportl(ui p , ui d)
{
    uint32 *ioport = (uint32 *)(caspian_iospace + p);
    *ioport = d & 0xFFFFFFFFU;
    DEBUG_IO("0x%X -> 0x%X\n", p, d );
    mb();
}

void outportq(ui p , ul d)
{
    uint64 *ioport = (uint64 *)(caspian_iospace + p);
    *ioport = d;
    DEBUG_IO("0x%X -> 0x%lX\n", p, d );
    mb();
}

ui inportb(ui p)
{
    uint8 *ioport = caspian_iospace + p;
    DEBUG_IO( "port 0x%X\n", p );
    return *ioport & 0xFFU;
}

ui inportw(ui p)
{
    uint16 *ioport = (uint16 *)(caspian_iospace + p);
    DEBUG_IO( "port 0x%X\n", p );
    return *ioport & 0xFFFFU;
}

ui inportl(ui p)
{
    uint32 *ioport = (uint32 *)(caspian_iospace + p);
    DEBUG_IO( "port 0x%X\n", p );
    return *ioport & 0xFFFFFFFFU;
}

ul inportq(ui p)
{
    uint64 *ioport = (uint64 *)(caspian_iospace + p);
    DEBUG_IO( "port 0x%lX\n", p );
    return *ioport;
}


/*--------------------------------------------------------------------*/
/* Low level PCI memory space access mechanisms */


static uint8 *caspian_memspace = (uint8 *)PCI_MEM;

void outmemb(ui p , ui d)
{
    uint8 *memport = caspian_memspace + p;
    *memport = d & 0xFFU;
    DEBUG_IO("0x%X -> 0x%X\n", p, d );
    mb();
}

void outmemw(ui p , ui d)
{
    uint16 *memport = (uint16 *)(caspian_memspace + p);
    *memport = d & 0xFFFFU;
    DEBUG_IO("0x%X -> 0x%X\n", p, d );
    mb();
}

void outmeml(ui p , ui d)
{
    uint32 *memport = (uint32 *)(caspian_memspace + p);
    *memport = d & 0xFFFFFFFFU;
    DEBUG_IO("0x%X -> 0x%X\n", p, d );
    mb();
}

void outmemq(ui p , ul d)
{
#ifdef DEBUG_GOBY_64BIT
    outmeml( p, (ui)d );
    outmeml( p+4, (ui)(d>>32) );
#else
    uint64 *memport = (uint64 *)(caspian_memspace + p);
    *memport = d;
    DEBUG_IO("0x%X -> 0x%lX\n", p, d );
    mb();
#endif
}


ui inmemb(ui p)
{
    uint8 *memport = caspian_memspace + p;
    DEBUG_IO( "port 0x%X\n", p );
    return *memport & 0xFFU;
}

ui inmemw(ui p)
{
    uint16 *memport = (uint16 *)(caspian_memspace + p);
    DEBUG_IO( "port 0x%X\n", p );
    return *memport & 0xFFFFU;
}

ui inmeml(ui p)
{
    uint32 *memport = (uint32 *)(caspian_memspace + p);
    DEBUG_IO( "port 0x%X\n", p );
    return *memport & 0xFFFFFFFFU;
}

ul inmemq(ui p)
{
#ifdef DEBUG_GOBY_64BIT
    uint64 val;
    val = (uint64)inmeml( p );
    mb();
    val |= (uint64)inmeml( p+4 ) << 32;
    return val;
#else
    uint64 *memport = (uint64 *)(caspian_memspace + p);
    DEBUG_IO( "port 0x%X\n", p );
    return *memport;
#endif
}



/*--------------------------------------------------------------------*/
/* Low level PCI configuration space access mechanisms */

#define PCI_CSR_ADDR(bus,dev,fun,reg)	\
	( (ul) (PCI_CSR | (reg & 0xFF) \
			| ((fun & 0x7) << 8) \
			| ((dev & 0x1F) << 11 ) \
			| ((bus & 0xFF) << 16 ) ) )

void pcicfgwb(ui bus , ui dev , ui func , ui reg , ui data)
{
    uint8 *csr = (uint8 *)PCI_CSR_ADDR(bus,dev,func,reg);
    *csr = data & 0xFFU;
    DEBUG_IO( "0x%X 0x%X 0x%X 0x%X -> 0x%X\n", bus, dev, func, reg, data );
    mb();
}

void pcicfgww(ui bus , ui dev , ui func , ui reg , ui data)
{
    uint16 *csr = (uint16 *)PCI_CSR_ADDR(bus,dev,func,reg);
    *csr = data & 0xFFFFU;
    DEBUG_IO( "0x%X 0x%X 0x%X 0x%X -> 0x%X\n", bus, dev, func, reg, data );
    mb();
}

void pcicfgwl(ui bus , ui dev , ui func , ui reg , ui data)
{
    uint32 *csr = (uint32 *)PCI_CSR_ADDR(bus,dev,func,reg);
    *csr = data & 0xFFFFFFFFU;
    DEBUG_IO( "0x%X 0x%X 0x%X 0x%X -> 0x%X\n", bus, dev, func, reg, data );
    mb();
}


ui pcicfgrb(ui bus , ui dev , ui func , ui reg)
{
    uint8 *memport = (uint8 *)PCI_CSR_ADDR(bus,dev,func,reg);
    DEBUG_IO( "0x%X 0x%X 0x%X 0x%X\n", bus, dev, func, reg );
    return *memport & 0xFFU;
}

ui pcicfgrw(ui bus , ui dev , ui func , ui reg)
{
    uint16 *memport = (uint16 *)PCI_CSR_ADDR(bus,dev,func,reg);
    DEBUG_IO( "0x%X 0x%X 0x%X 0x%X\n", bus, dev, func, reg );
    return *memport & 0xFFFFU;
}

ui pcicfgrl(ui bus , ui dev , ui func , ui reg)
{
    uint32 *memport = (uint32 *)PCI_CSR_ADDR(bus,dev,func,reg);
    DEBUG_IO( "0x%X 0x%X 0x%X 0x%X\n", bus, dev, func, reg );
    return *memport & 0xFFFFFFFFU;
}


/*--------------------------------------------------------------------*/
/* Miscellaneous PCI supporting routines */

ui inIack(void )
{
    uint8 *iack = (uint8 *)PCI_IACK;
    return *iack & 0xFFU;
}

ul IOPCIClearNODEV( void )
{
    return FALSE;
}

void outled( unsigned val )			/* diagnostic data */
{
}




/*--------------------------------------------------------------------*/
/* Fixups of the Northbridge that aren't covered by the normal init path */

void nb_setup( void )
{
    TRACE("Running...\n");

    /* setup the PCI and AGP interfaces */
    diags_subsys_stat( SYS_PCI, DEV_PROBING );

    PCIInit();

    diags_subsys_stat( SYS_PCI, DEV_SETUP );
}

void nb_fixup( void )
{
    TRACE("Running...\n");
}


/* enable ECC reporting, correction in the Northbridge - 1=enable, 0=disable */
void nb_ecc( unsigned ena )
{
}

int nb_eccstat( char *buf )		/* 0 if OK, -1 if error occurred */
{
    return 0;
}


/*----------------------------------------------------------------------*/
/* SMP functionality */

unsigned nb_whami( void )		/* my CPU ID, starting from 0 */
{
    return pcicfgrb( 0, GOBY0_DEV, 0, GOBY0_WHAMI );
}



/*----------------------------------------------------------------------*/
/* Caspian-specific interface routines */

void goby_revstr( char *buf )
{
    unsigned char revcode, rev, stepping;

    revcode = pcicfgrb( 0, GOBY0_DEV, 0, GOBY0_REV );
    rev = revcode >> 4;
    stepping = revcode & 0xF;
    sprintf_dbm( buf, "Revision %c stepping %d", rev + 'A', stepping + 1 );
}



/*
 * Interrupt management routines
 * Clearly, the Goby interrupt system is not derived from Irongate...
 * 
 * Goby interrupt operation:
 * -------------------------
 * We have a set of PCI-memory-mapped CSRs, the base of which is given by
 * GOBY0_BAR3 (0x1C).
 *
 * This region is divided into three parts, one for common configuration 
 * and one for each system port, starting at 0x0, 0x100, 0x200 respectively
 * 
 * Common configuration registers:
 * -------------------------------
 * IRR          Interrupt Requests Received                     Offset 0x0
 * 63-0         Interrupt requests as raw input.
 *              NOTE: what decides, for example, what interrupt bit 57 is?
 *              -> motherboard wiring?
 *
 * ITM          Interrupt Trigger Mode                          Offset 0x20
 * 63-0         The trigger mode (0=edge, 1=level) for each IRR bit.
 *
 * ILR          Interrupt Line Routing                          Offset 0x40
 * 191-0        Split into 64 3-bit fields.  Each 3-bit field contains an
 *              Alpha IRQ number (0-5, 6-7 reserved) for a corresponding IRR
 * 
 * LIR          Local interrupt routing                         Offset 0xA0
 * 63           Periodic timer interrupt enable (1=enable)
 * 57           Periodic timer interrupt mode (always 0=edge triggered)
 * 56           Periodic timer interrupt polarity (always 1=active low)
 * 55-48        Periodic timer interrupt vector (fixed at 0x33?)
 * 47           NMI enable (1=enable)
 * 41           NMI mode (always 0=edge)
 * 40           NMI polarity (always 0=high)
 * 39-32        NMI vector (fixed at 0x32?)
 * 31           SMI enable (1=enable)
 * 25           SMI mode (always 0=edge)
 * 24           SMI polarity (always 1=low)
 * 23-16        SMI vector (fixed at 0x31?)
 * 15           Local IO interrupt enable (1=enable)
 * 9            Local IO interrupt mode (always 0=edge)
 * 8            Local IO interrupt polarity (always 0=high)
 * 7-0          Local IO interrupt vector (fixed at 0x30?)
 *
 * EIR          Error interrupt routing                         Offset 0xA8
 * 31           Correctable Error enable (1=enable)
 * 25           Correctable Error interrupt mode (always 1=level)
 * 24           Correctable error interrupt polarity (always 0=high)
 * 23-16        Correctable error interrupt vector (fixed at 0x35?)
 * 15           Fatal error enable (1=enable)
 * 9            Fatal error interrupt mode (always 1=level)
 * 8            Fatal error interrupt polarity (always 0=high)
 * 7-0          Fatal error interrupt vector (always reads 0x34?)
 *
 * Per-processor configuration registers:
 * --------------------------------------
 * IRPx         Interrupt Requests Pending (port x)     Offset 0x100, 0x200
 * 63-0         Interrupts awaiting servicing on port x
 *
 * ISRx         Interrupt Service Requests (port x)     Offset 0x120, 0x220
 * 63-0         A CPU sets a bit in this register to acknowledge an interrupt
 *              request (a bit set in the IRPx and IRR).  A CPU clears a bit
 *              in this register by writing the vector number (0-63?) to the 
 *              EOI register.  Whilst a bit is set in this register, all the 
 *              corresponding interrupt activity is sent to this processor.
 *
 * IEMx         Interrupt enables mask (port x)         Offset 0x140, 0x240
 * 63-0         A bit set in this register enables an interrupt to go to the
 *              processor.  Otherwise the interrupt request is ignored.
 *
 * EOIx         End of interrupt (port x)               Offset 0x160, 0x260
 * 19           Activate EOI - send off all EOIs recorded into LDT-world...
 * 7-0          EOI vector number.
 *              NOTE: why 8 bits when there are 64 interrupt sources??
 *
 * SIRx         Send interrupt request (port x)         Offset 0x180, 0x280
 * 19           Activate - when a 1 is written things happen, otherwise not?
 * 18           Trigger mode - 0=edge, 1=level - of int to be delivered
 * 16           Dest mode???
 * 15-8         Destination ID for interrupt to be delivered
 * 7-0          Interrupt source bit number (vector).
 * 
 * IIx          Interrupt identification (port x)       Offset 0x188, 0x288
 * 23-16        Logical ID mask
 * 15-8         Logical ID
 * 7-0          Physical ID
 *              NOTE: what is going on here???
 *
 * ITPx         Interrupt task priority (port x)        Offset 0x190, 0x290
 * 7-0          Task priority of this system port.  Always zero.
 */
/*------------------------------------------------------------------------*/


/* Dump the IO-space mapped interrupt CSRs */
#define ILR_BYTES (192/8)

static unsigned ilr_get( const unsigned base, const int irq )
{
    uint8 byte1, byte2;			/* offsets into ILR to read from */
    unsigned bitshift;			/* bit-shift into first byte */
    unsigned value;

    byte1 = (3 * irq) / 8;
    byte2 = (3 * irq + 2) / 8;
    bitshift = (3 * irq) % 8;

    if ( byte1 == byte2 )		/* process one byte or two? */
	value = inmemb( base + GOBY_IRQ_ILR + byte1 );
    else
	value =  inmemb( base + GOBY_IRQ_ILR + byte1 ) | 
		(inmemb( base + GOBY_IRQ_ILR + byte2 ) << 8);

    return (value >> bitshift) & 0x7;
}

static void ilr_set( const unsigned base, const int irq, const unsigned val )
{
    uint8 byte1, byte2;                 /* offsets into ILR to read from */
    unsigned bitshift;                  /* bit-shift into first byte */
    unsigned mask;
    unsigned value;

    byte1 = (3 * irq) / 8;
    byte2 = (3 * irq + 2) / 8;
    bitshift = (3 * irq) % 8;
    mask = 0x7 << bitshift;

    if ( byte1 == byte2 )               /* process one byte or two? */
        value = inmemb( base + GOBY_IRQ_ILR + byte1 );
    else
        value =  inmemb( base + GOBY_IRQ_ILR + byte1 ) |
                (inmemb( base + GOBY_IRQ_ILR + byte2 ) << 8);

    value &= ~mask;
    value |= val << bitshift;
    
    if ( byte1 == byte2 )               /* process one byte or two? */
    {
	outmemb( base + GOBY_IRQ_ILR + byte1, value );
    } else {
	/* here we rely on outmemb truncating to write only one byte */
	outmemb( base + GOBY_IRQ_ILR + byte1, value );
	outmemb( base + GOBY_IRQ_ILR + byte2, value >> 8 );
    }
}



static void goby_irq_dump_common( unsigned base )
{
    int i, j;

    mobo_logf(	LOG_DBG "Common Interrupt CSRs at 0x%X\n", base );
    mobo_logf(	LOG_DBG "  0x%016lX : Interrupt Requests Received\n"
		LOG_DBG "  0x%016lX : Interrupt Trigger Mode\n"
		LOG_DBG "  0x%016lX : Local Interrupt Routing\n"
		LOG_DBG "  0x%016lX : Error Interrupt Routing\n",
		inmemq( base + GOBY_IRQ_IRR ),
		inmemq( base + GOBY_IRQ_ITM ),
		inmemq( base + GOBY_IRQ_LIR ),
		inmemq( base + GOBY_IRQ_EIR ) );


    /* Take a look at all the interrupt routings */
    mobo_logf(	LOG_DBG "Goby IRQ <-> Alpha hardware interrupt routings:\n" );

    mobo_logf(  LOG_DBG "Raw value: 0x%08X%08X%08X%08X%08X%08X\n",
		inmeml( base + GOBY_IRQ_ILR + 20 ),
		inmeml( base + GOBY_IRQ_ILR + 16 ),
		inmeml( base + GOBY_IRQ_ILR + 12 ),
		inmeml( base + GOBY_IRQ_ILR + 8 ),
		inmeml( base + GOBY_IRQ_ILR + 4 ),
		inmeml( base + GOBY_IRQ_ILR ) );

    mobo_logf(	LOG_DBG "    " );
    for ( i=0; i<16; i++ )
	mobo_logf( "%02d ", i );
    mobo_logf( "\n" );

    for ( i=0; i<64; i+=16 )
    {
	mobo_logf( LOG_DBG "%02d: ", i );
	for ( j=0; j<16; j++ )
	    mobo_logf( "%02d ", ilr_get( base, i+j ) );
	mobo_logf( "\n" );
    }
}

static void goby_irq_dump_cpu( unsigned cpubase )
{
    mobo_logf(	LOG_DBG "Port specific CSRs at 0x%X\n", cpubase );
    mobo_logf(	LOG_DBG "  0x%016lX : Interrupt Requests Pending\n"
		LOG_DBG "  0x%016lX : Interrupt Service Requests\n"
		LOG_DBG "  0x%016lX : Interrupt Enables Mask\n"
		LOG_DBG "  0x%016lX : Send Interrupt Request\n"
		LOG_DBG "  0x%016lX : Interrupt Identification\n",
		inmemq( cpubase + I_IRP ),
		inmemq( cpubase + I_ISR ),
		inmemq( cpubase + I_IEM ),
		inmemq( cpubase + I_SIR ),
		inmemq( cpubase + I_II ) );
}

void goby_irq_dump( void )
{
    unsigned base = goby_irq_base();
    TRACE( "Examining Goby IRQ CSRs\n" );
    goby_irq_dump_common( base );			/* Common registers */
    goby_irq_dump_cpu( base + I_PORT0 );		/* System port 0 */
    goby_irq_dump_cpu( base + I_PORT1 );		/* System port 1 */
}


/* Return the base address of the interrupt CSRs */
unsigned goby_irq_base( void )
{
    return pcicfgrl( 0, 0, 0, GOBY0_IRQ ) & ~0xFU;
}


/* Set a bit in the interrupt routings to make a map between goby and alpha */
/* This is done using the mammoth Interrupt Line Routing (ILR) register */
void goby_map_irq( const unsigned goby_irq, const unsigned alpha_irq )
{
    if ( goby_irq < 0 || goby_irq > 63 )
    {
	mobo_alertf( "internal error",
		"Tried to map a goby irq (%d) that was not 0-63", goby_irq );
	return;
    }

    if ( alpha_irq < 0 || alpha_irq > 5 )
    {
	mobo_alertf( "internal error",
		"Tried to map an alpha irq (%d) that was not 0-5", alpha_irq );
	return;
    }

    TRACE( "mapping Goby IRQ %d to Alpha IRQ %d...\n", goby_irq, alpha_irq );
    ilr_set( goby_irq_base(), goby_irq, alpha_irq );
}


/* Unmask (enable) a particular goby irq on a particular processor */
void goby_enable_irq( const unsigned goby_irq, const unsigned cpuid )
{
    uint64 irq_mask;
    unsigned irq_addr;

    irq_addr = goby_irq_base();
    if (cpuid==0)
	irq_addr += GOBY_IRQ_IEM0;
    else
	irq_addr += GOBY_IRQ_IEM1;


    TRACE( "Enabling IRQ %d on CPU %d (reading port %08X)...\n", 
		goby_irq, cpuid, irq_addr );

    irq_mask = inmemq( irq_addr );
    irq_mask |= 1UL << goby_irq;
    outmemq( irq_addr, irq_mask );

    TRACE( "CPU %d Interrupt Enable Mask is now 0x%016lX (want 0x%016lX)\n", 
	cpuid, inmemq( irq_addr ), irq_mask );
}


/* Send a given interrupt to a given processor */
#define SEND_TIMEOUT		10

void goby_send_irq( const unsigned goby_irq, const unsigned cpuid )
{
    uint64 irq_mask;
    uint64 send_req;
    unsigned cpu_addr;
    int i;

    TRACE( "Sending IRQ %d to CPU %d...\n", goby_irq, cpuid );
    cpu_addr = goby_irq_base() + (cpuid==0) ? I_PORT0 : I_PORT1;

    irq_mask = inmemq( cpu_addr + I_IEM );
    if ( (irq_mask & (1UL << goby_irq)) == 0UL )
    {
	mobo_logf( LOG_WARN "CPU %d wasn't listening for IRQ %d, enabling...\n",
		cpuid, goby_irq );
	goby_enable_irq( goby_irq, cpuid );
    }

    send_req = SIR_ACT | SIR_TRIG_LV |
		(cpuid << SIR_DEST_S) | (goby_irq << SIR_IRQ_S);

    TRACE( "Sending 0x%016lX & waiting for 'activate' to clear.", send_req );
    outmemq( cpu_addr + I_SIR, send_req );

    for( i=0; i<SEND_TIMEOUT; i++ )
    {
    	send_req = inmemq( cpu_addr + I_SIR );
	if ( send_req & SIR_ACT )
	{
	    TRACE( "Interrupt delivered (SIR is 0x%016lX)!\n", send_req );
	    break;
	}

	TRACE( "SIR = 0x%016lX, interrupt not yet delivered...\n", send_req );
    }

}




