/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* IronGate management library, (c) 1999 Alpha Processor, Inc.
 * Begun 19 January 1999 by Stig Telfer, Alpha Processor, Inc.
 */

#include "lib.h"
#include "uilib.h"
#include "info.h"
#include "pci.h"
#include "memory.h"
#include "ledcodes.h"

#include "uilib.h"
#include "northbridge.h"


/*----------------------------------------------------------------------*/
/* Public visible Irongate configuration data */



/* number of independent PCI bus trees */
const unsigned char nb_pci_hosecount=1; 


/*----------------------------------------------------------------------*/
/* IronGate system memory map */

#define PCI0_MEMORY     0x80000		/* PCI Memory space on bus 0 */
#define PCI0_IACK       0x801F8		/* Interrupt acknowledge cycles */
#define PCI0_IO         0x801FC		/* IO space */
#define PCI0_CONFIG     0x801FE		/* Configuration space */

/* full 64-bit addresses */
#define PCI_MEM   (((ul) PCI0_MEMORY<<24) | DBM_SUPER )	
#define IACK      (((ul) PCI0_IACK<<24) | DBM_SUPER )
#define PCI_IO    (((ul) PCI0_IO<<24)   | DBM_SUPER )
#define PCI_CFG   (((ul) PCI0_CONFIG<<24) | DBM_SUPER )


/*----------------------------------------------------------------------*/
/* CSR managment code */

typedef int (*ig_csr2str)(reg_handle, char * );

#define RO 1
#define WO 2
#define RW 3

typedef struct {
	ig_csr2str	pfn;		/* function to print */
	uint8		mode;		/* readable, writeable etc. */
	uint8		offset;		/* offset in the IG CSR space */
	uint8		size;		/* size of CSR in bytes */
} ig_csrdesc;

/* function prototypes */
static int p_std8( reg_handle pval, char *buf);
static int p_std16( reg_handle pval, char *buf);
static int p_std32( reg_handle pval, char *buf);
static int p_command( reg_handle pval, char *buf);
static int p_status( reg_handle pval, char *buf);
static int p_revid( reg_handle pval, char *buf);
static int p_bar0( reg_handle pval, char *buf);
static int p_bar1( reg_handle pval, char *buf);
static int p_bacsr( reg_handle pval, char *buf);
static int p_ecc( reg_handle pval, char *buf);
static int p_dram_ms( reg_handle pval, char *buf);


/* big IronGate CSR description vector */
static const ig_csrdesc ig_csr[] = {
	{ p_std16, RO, 0x00, 2 },	/* IG_VENDORID */
	{ p_std16, RO, 0x02, 2 },	/* IG_DEVICEID */
	{ p_command, RW, 0x04, 2 },	/* IG_COMMAND */
	{ p_status, RO, 0x06, 2 },	/* IG_STATUS */
	{ p_revid, RO, 0x08, 1 },	/* IG_REVID */
	{ p_std8, RO, 0x09, 1 },	/* IG_PROGIF */
	{ p_std8, RO, 0x0A, 1 },	/* IG_SUBCLASS */
	{ p_std8, RO, 0x0B, 1 },	/* IG_BASECLASS */
					/* offset 0x0C RESERVED */
	{ p_std8, RO, 0x0D, 1 },	/* IG_LATENCY */
	{ p_std8, RO, 0x0E, 1 },	/* IG_HEADER */
	{ p_bar0, RW, 0x10, 4 },	/* IG_BAR0 */
	{ p_bar1, RW, 0x14, 4 },	/* IG_BAR1 */
					/* Lots of other stuff in here */
	{ p_bacsr, RW, 0x40, 2 },	/* IG_BACSR0 */
	{ p_bacsr, RW, 0x42, 2 },	/* IG_BACSR1 */
	{ p_bacsr, RW, 0x44, 2 },	/* IG_BACSR2 */
	{ p_bacsr, RW, 0x46, 2 },	/* IG_BACSR3 */
	{ p_bacsr, RW, 0x48, 2 },	/* IG_BACSR4 */
	{ p_bacsr, RW, 0x4A, 2 },	/* IG_BACSR5 */
					/* More stuff in here */
	{ p_ecc, RW, 0x58, 2 },		/* IG_ECC */
	{ p_dram_ms, RW, 0x5A, 2 },	/* IG_DRAM_MS */
					/* More stuff in here */
	{ p_std32, RW, 0x84, 4 },	/* IG_PCI_ARB */
};

#define CFGADDR	0x0CF8
#define CFGDATA 0x0CFC

#define BUS0	0
#define IG	0
#define FN0	0

reg_handle ig_rdcsr(enum ig_csr csr, reg_handle rval)
{
	uint8 reg = ig_csr[csr].offset;

	usleep( 1000 );			/* DEBUG delay - Irongate needs this? */
	switch( ig_csr[csr].size )
	{
	    case 1:
		*rval.p8 = pcicfgrb(BUS0, IG, FN0, reg );
		break;

	    case 2:
		*rval.p16 = pcicfgrw(BUS0, IG, FN0, reg );
		break;

	    case 4:
		*rval.p32 = pcicfgrl(BUS0, IG, FN0, reg );
		break;
	}
	usleep( 1000 );			/* DEBUG delay - Irongate needs this? */

	return rval;
}



int ig_wrcsr(enum ig_csr csr, reg_handle wval)
{
	uint8 reg = ig_csr[csr].offset;

	asm ( "mb" );			/* defensive coding */
	usleep( 1000 );			/* DEBUG delay - Irongate needs this? */
	switch( ig_csr[csr].size )
	{
	    case 1:
		pcicfgwb( BUS0, IG, FN0, reg, *wval.p8 );
		break;

	    case 2:
		pcicfgww( BUS0, IG, FN0, reg, *wval.p16 );
		break;

	    case 4:
		pcicfgwl( BUS0, IG, FN0, reg, *wval.p32 );
		break;
	}

	asm ( "mb" );
	usleep( 1000 );			/* DEBUG delay - Irongate needs this? */

	return 0;
}

int ig_pcsr(enum ig_csr csr, reg_handle pval, char *buf)
{
	return ig_csr[csr].pfn( pval, buf );
}



/*----------------------------------------------------------------------*/
/* Private routines */

static int p_std8( reg_handle pval, char *buf)
{
	return sprintf_dbm(buf, "0x%02X", *pval.p8 );
}

static int p_std16( reg_handle pval, char *buf)
{
	return sprintf_dbm(buf, "0x%04X", *pval.p16 );
}

static int p_std32( reg_handle pval, char *buf)
{
	return sprintf_dbm(buf, "0x%08X", *pval.p32 );
}

static int p_command( reg_handle pval, char *buf)
{
	Command_t val;
	val.i = *pval.p16;	/* initialise union via uint16 */
	return sprintf_dbm(buf, "0x%04X -> %s %s %s %s %s %s %s %s %s %s",
			val.i,
			val.r.fbbce ? "FBBCE" : "fbbce",
			val.r.serre ? "SERRE" : "serre",
			val.r.step ? "STEP" : "step",
			val.r.per ? "PER" : "per",
			val.r.vgaps ? "VGAPS" : "vgaps",
			val.r.mwic ? "MWIC" : "mwic",
			val.r.scmon ? "SCMON" : "scmon",
			val.r.iten ? "ITEN" : "iten",
			val.r.memspc ? "MEMSPC" : "memspc",
			val.r.iospc ? "IOSPC" : "iospc" );
}

static int p_status( reg_handle pval, char *buf)
{
	Status_t val;
	static const char *devsel_timing[4] = { "!rsvd", "!slow", "MED", "!fast" };
	val.i = *pval.p16;
        return sprintf_dbm(buf, "0x%04X -> %s %s %s %s %s %s %s %s %s %s %s",
			val.i,
			val.r.dpe ? "DPE" : "dpe",
			val.r.serr ? "SERR" : "serr",
			val.r.ria ? "RIA" : "ria",
			val.r.rta ? "RTA" : "rta",
			val.r.sta ? "STA" : "sta",
			devsel_timing[val.r.devsel],
			val.r.ppe ? "PPE" : "ppe",
			val.r.fbbc ? "FBBC" : "fbbc",
			val.r.udf ? "UDF" : "udf",
			val.r.pci66 ? "66MHz" : "33MHz",
			val.r.cl ? "CL" : "cl" );
}

static int p_revid( reg_handle pval, char *buf)
{
	RevID_t val;
	val.i = *pval.p8;
        return sprintf_dbm(buf, "0x%02X -> Rev %c stepping %d",
			val.i, 'A' + val.r.die, val.r.step );
}

static int p_bar0( reg_handle pval, char *buf)
{
        return sprintf_dbm(buf, "0x%04X", *pval.p16 );
}

static int p_bar1( reg_handle pval, char *buf)
{
        return sprintf_dbm(buf, "0x%04X", *pval.p16 );
}

static int p_bacsr( reg_handle pval, char *buf)
{
	BACSR_t val;
	unsigned banksize, bankbase;
	unsigned i, msk;

	val.i = *pval.p16;
	bankbase = (unsigned)val.r.base << 3;	/* base addr in Mbytes */
	for (banksize = 8, msk = val.r.mask, i = 1; i <= 6; i++)
	{
		if ( msk & 0x1 ) 	banksize += 8 << i;
		msk = msk >> 1;
	}

	return sprintf_dbm(buf, "0x%04X -> bank %s, base %dM size %dM",
			val.i, 
			val.r.en ? "ENABLED" : "disabled",
			bankbase, banksize );
}

static int p_ecc( reg_handle pval, char *buf)
{
	int bankno, csdata;
	static const char *ECC_status[] = { "No Error",
					"Multi-bit Err", 
					"Single-bit Err",
					"Multiple Errs" };
	ECC_t val;
	val.i = *pval.p16;

	sprintf_dbm( buf, "0x%04X -> ", val.i );
	buf += strlen( buf );				/* to end of string */

	/* if the chipselects are zero, and an error is signalled, it's NXM */
	if ( val.r.status != 0 ) {
	    if ( val.r.chipsel != 0 )
	    {
		csdata = val.r.chipsel, bankno=0;
		while ( (csdata & 0x1) == 0 ) 	bankno++, csdata >>= 1;

		sprintf_dbm( buf, "%s chipselect %d",
			ECC_status[val.r.status], bankno );
	    }
	    else strcat( buf, "Non-existent memory access" );
	}
	else strcat( buf, "No Error" );

 	return 0;
}

static int p_dram_ms( reg_handle pval, char *buf)
{
	static const uint16 rcycles[] = { 2048, 1536, 1024, 512 };
	DRAM_MS_t val;
	val.i = *pval.p16;
	return sprintf_dbm(buf, "0x%04X -> %s %s %s %s %c %s %d cycles/refresh",
		val.i,
		val.r.sdraminit ? "SINIT" : "sinit",
		val.r.type ? "!ESDRAM" : "SDRAM",
		val.r.mwe ? "MWE" : "mwe",
		val.r.bre ? "BRE" : "bre",
		val.r.r ? 'R' : 'r',
		val.r.en ? "EN" : "en",
		rcycles[val.r.cycles] );
}



/*----------------------------------------------------------------------*/
/* Control/Analyse the Irongate CSRs for SERR/ECC error conditions */

void nb_ecc( unsigned o )		/* o = 'on' or 'off' */
{
    DRAM_MS_t DRAM_MS;
    reg_handle r;
    char strbuf[64];


    r.p16 = &DRAM_MS.i;
    ig_rdcsr( IG_DRAM_MS, r );


    if ( o )
	DRAM_MS.r.en = 1;                    /* enable the appropriate bit */
    else
	DRAM_MS.r.en = 0;
    
    ig_pcsr( IG_DRAM_MS, r, strbuf );
    ig_wrcsr( IG_DRAM_MS, r );
}


#if 0			/* NB this function is DISUSED: soon to disappear? */

int nb_eccstat( char *buf )
{
    ECC_t ECC;                          /* Irongate ECC register */
    Status_t Status;
    Command_t Command;
    reg_handle r, s, c;
    unsigned eccval;

    /* read the ECC register */
    r.p16 = &ECC.i;
    ig_rdcsr( IG_ECC, r );
    ig_pcsr( IG_ECC, r, buf );

    eccval = ECC.r.status;
    if ( eccval )			/* ie an error has been logged */
    {
	/* to enable further ECC errors, the bits must be cleared */
	/* Any SERR must be acknowledged (if it hasn't triggered NMI) */
	Status.i = 0x4000;		/* magic: clears the SERR bit */
	s.p16 = &Status.i;
	ig_wrcsr( IG_STATUS, s );

	/* I suspect the SERRE bit in the Command register needs setting */
	c.p16 = &Command.i;
	ig_rdcsr( IG_COMMAND, c );
	Command.r.serre = 1;
	ig_wrcsr( IG_COMMAND, c );

	ECC.r.status = 0x3;		/* seem to be W1C, not W0C? */
	ig_wrcsr( IG_ECC, r );

    }
    return eccval;
}

#endif					/* 0 */


/*----------------------------------------------------------------------*/
/* nb_setup - early initialisations relating to the northbridge. */ 

void nb_setup( void )
{
    Status_t Status;
    Command_t Command;
    PCI_arb_t PCI_arb;
    reg_handle r;


    /* Enable those magic XCA bits that stop the system hanging */

    pcicfgwb( 0, 0, 0, 0x62, 0xA0 );
    pcicfgwb( 0, 0, 0, 0x63, 0x85 );
    

    /* enable ECC registers, SERR reporting */
    nb_ecc( 1 );

    r.p16 = &Status.i;
    ig_rdcsr( IG_STATUS, r );
    Status.r.serr = 0;			/* clear any outstanding SERRs */
    ig_wrcsr( IG_STATUS, r );
    r.p16 = &Command.i;			/* find CSR with SERR enabling */
    ig_rdcsr( IG_COMMAND, r );
    Command.r.serre = 1;
    ig_wrcsr( IG_COMMAND, r );		/* write it back */


    /* set DMA mode to EV6 in PCI arbitration control register */

    r.p32 = &PCI_arb.i;
    ig_rdcsr( IG_PCI_ARB, r );
    PCI_arb.r.ev6=1;
    ig_wrcsr( IG_PCI_ARB, r );


    /* setup the PCI and AGP interfaces */
    diags_subsys_stat( SYS_PCI, DEV_PROBING );

    PCIInit();

    diags_subsys_stat( SYS_PCI, DEV_SETUP );
}



/*----------------------------------------------------------------------*/
/* nb_fixup - late initialisations and fixups */

void nb_fixup( void )
{
    char IG_rev[ 32 ];
    RevID_t RevID;
    reg_handle r;

    /* Register information about this Irongate */
    r.p8 = &RevID.i;
    ig_rdcsr( IG_REVID, r );
    ig_pcsr( IG_REVID, r, IG_rev );
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "AMD Irongate Revision", IG_rev );
}




/*------------------------------------------------------------------------**
**                                                                        **
**  DP264 I/O procedures                                                  **
**                                                                        **
**      inport[b|w|l], outport[b|w|l] 8:16:32 IO xfers                    **
**                                                                        **
**      inmem[b|w|l], outmem[b|w|l] 8:16:32 PCI memory xfers              **
**                                                                        **
**------------------------------------------------------------------------*/



#define QW(d)	((ul)(d))

#define SWZ(d) ( \
		(((QW(d)>>(0*8))&0xff)<<(0*8)) | \
		(((QW(d)>>(1*8))&0xff)<<(2*8)) | \
		(((QW(d)>>(2*8))&0xff)<<(4*8)) | \
		(((QW(d)>>(3*8))&0xff)<<(6*8)) | \
		(((QW(d)>>(4*8))&0xff)<<(1*8)) | \
		(((QW(d)>>(5*8))&0xff)<<(3*8)) | \
		(((QW(d)>>(6*8))&0xff)<<(5*8)) | \
		(((QW(d)>>(7*8))&0xff)<<(7*8)))

#define UNSWZ(d) ( \
		(((QW(d)>>(0*8))&0xff)<<(0*8)) | \
		(((QW(d)>>(2*8))&0xff)<<(1*8)) | \
		(((QW(d)>>(4*8))&0xff)<<(2*8)) | \
		(((QW(d)>>(6*8))&0xff)<<(3*8)) | \
		(((QW(d)>>(1*8))&0xff)<<(4*8)) | \
		(((QW(d)>>(3*8))&0xff)<<(5*8)) | \
		(((QW(d)>>(5*8))&0xff)<<(6*8)) | \
		(((QW(d)>>(7*8))&0xff)<<(7*8)))

static void WriteIOB(ul address,ub value) {WriteB((address), (value)); }
static void WriteIOW(ul address,uw value) {WriteW((address), (value)); }
static void WriteIOL(ul address,ui value) {WriteL((address), (value)); }

static ub ReadIOB(ul address) {  return(ReadB(address)); }
static uw ReadIOW(ul address) {  return(ReadW(address)); }
static ui ReadIOL(ul address) {  return(ReadL(address)); }


/*
 * The following macro define Type 0 and Type 1 configuration address format
 *
 *  4 4 4 4|3 3 3 3|3 3 3 3|3 3 2 2|2 2 2 2|2 2 2 2|1 1 1 1|1 1 1 1|1 1 
 *  3 2 1 0|9 8 7 6|5 4 3 2|1 0 9 8|7 6 5 4|3 2 1 0|9 8 7 6|5 4 3 2|1 0 9 8|7 6 5 4|3 2 1 0
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * |1|0|0|0|0|0|0|0|0|0|x|      0x1FE      |       bus     |  device |func.|  register |   |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                                                                       |
 *                                                                               Byte ---+
 */

#define PCI_CONFIG(bus,device,function,register) \
	((ul)(PCI_CFG | (register & 0xff) | ((function & 0x7) << 8) \
	      | ((device & 0x1f) << 11) | ((bus & 0xff) << 16)))

#define IO_ACCESS(p) ((ul)(PCI_IO | p))
#define MEM_ACCESS(p) ((ul)(PCI_MEM | p))


void pcicfgwb(ui bus, ui dev, ui func, ui reg, ui data)
{
  data&=0xff; 
  mb(); /* DEBUG */
  WriteIOB(PCI_CONFIG(bus,dev,func,reg), data);
  /* DEBUG mb(); */
}
void pcicfgww(ui bus, ui dev, ui func, ui reg, ui data)
{
  data&=0xffff; 
  mb(); /* DEBUG */
  WriteIOW(PCI_CONFIG(bus,dev,func,reg), data);
  /* DEBUG mb(); */
}
void pcicfgwl(ui bus, ui dev, ui func, ui reg, ui data)
{
  mb(); /* DEBUG */
  WriteIOL(PCI_CONFIG(bus,dev,func,reg), data);
  /* DEBUG mb(); */
}

ui pcicfgrb(ui bus, ui dev, ui func, ui reg)
{
  ul result;
  result = ReadIOB(PCI_CONFIG(bus,dev,func,reg));
  return(result & 0xff);
}

ui pcicfgrw(ui bus, ui dev, ui func, ui reg)
{
  ul result;
  result = ReadIOW(PCI_CONFIG(bus,dev,func,reg));
  return((result) & 0xffff);
}

ui pcicfgrl(ui bus, ui dev, ui func, ui reg)
{
  ul result;
  result = ReadIOL(PCI_CONFIG(bus,dev,func,reg));
  return(result & 0xffffffff);
}

/* ----- port routines -------------*/


void outportb(ui port,ui data)
{  
  data&=0xffU;
  mb(); /* DEBUG */
  WriteIOB(IO_ACCESS(port),data);  
  /* DEBUG mb(); */
}

void outportw(ui port,ui data)
{
  data&=0xffffU;
  mb(); /* DEBUG */
  WriteIOW(IO_ACCESS(port),data);
  /* DEBUG mb(); */
}

void outportl(ui port,ui data)
{
  mb(); /* DEBUG */
  WriteIOL(IO_ACCESS(port),data);
  /* DEBUG mb(); */
}

ui inportb(ui port)
{
  ul result;
  result = ReadIOB(IO_ACCESS(port));
  return(result & 0xff);
}

ui inportw(ui port)
{
  ul result;
  result = ReadIOW(IO_ACCESS(port));
  return(result & 0xffff);
}

ui inportl(ui port)
{
  ul result;
  result = ReadIOL(IO_ACCESS(port));
  return(result & 0xffffffff);
}




/* ----- memory routines -------------*/

void outmemb(ui port,ui data)
{
  data&=0xff;
  mb(); /* DEBUG */
  WriteIOB(MEM_ACCESS(port),data);
  /* DEBUG mb(); */
}

void outmemw(ui port,ui data)
{
  data&=0xffff;
  mb(); /* DEBUG */
  WriteIOW(MEM_ACCESS(port),data);
  /* DEBUG mb(); */
}

void outmeml(ui port,ui data)
{
  mb(); /* DEBUG */
  WriteIOL(MEM_ACCESS(port),data);
  /* DEBUG mb(); */
}

ui inmemb(ui port)
{
  ub result;
  result = ReadIOB(MEM_ACCESS(port));
  return(result & 0xff);
}

ui inmemw(ui port)
{
  uw result;
  result = ReadIOW(MEM_ACCESS(port));
  return(result & 0xffffU);
}

ui inmeml(ui port)
{
  ui result;
  result = ReadIOL(MEM_ACCESS(port));
  return(result & 0xffffffffU);
}

/*------------------------------------------------*/

ui inIack(void)
{
    return (*((ui*)IACK) & 0xffUL);
}



/*
 * Clear the NODEV status and
 * return the current value (TRUE/FALSE).
 */
ul IOPCIClearNODEV(void)
{
  return FALSE;
}


/*----------------------------------------------------------------------*/
/* SMP functionality - Irongate systems are uniprocessor */

unsigned nb_whami( void ) { return 0; }


