/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* System-related configuration details for the API Nautilus platform */
/* Begun by Stig Telfer, Alpha Processor Inc, 28 Mar 1999 */

#include "lib.h"
#include "uilib.h"

#include "specifics.h"			/* platform specific definitions */
#include "platform.h"			/* interface for this module */
#include "northbridge.h"		/* Generic NB access */
#include "southbridge.h"		/* Generic SB access */

/*----------------------------------------------------------------------*/
/* Reading from a ROM appears to be a highly platform-dependent thing
 */

const unsigned long plat_romsize = 2UL << 20;

static rom_t nautilus_nvram = {

    0,							/* hstart */
    0x1FB800U, 0x800U,					/* rstart, rsize */
    0x1FA000U, 0x2000U,					/* astart, asize */

    ROM_IMGVALID | ROM_PLATRSVD,			/* flags */
    FW_NVRAM,						/* fwid */
    
    "NVRAM data",					/* name */
    FG_CYAN,						/* colour */

    NULL,						/* hdr */
    NULL,						/* body */
    NULL						/* next */
};

static rom_t nautilus_nvlog = {

    0,							/* hstart */
    0x1F0000U, 0x8000U,					/* rstart, rsize */
    0x1F0000U, 0x8000U,					/* astart, asize */

    ROM_IMGVALID | ROM_PLATRSVD,			/* flags */
    FW_NVLOG,						/* fwid */
    
    "Diags log",					/* name */
    FG_CYAN,						/* colour */

    NULL,						/* hdr */
    NULL,						/* body */
    &nautilus_nvram					/* next */
};

static rom_t nautilus_nvenv = {

    0,							/* hstart */
    0x1F8000U, 0x2000U,					/* rstart, rsize */
    0x1F8000U, 0x2000U,					/* astart, asize */

    ROM_IMGVALID | ROM_PLATRSVD,			/* flags */
    FW_NVENV,						/* fwid */
    
    "Diags env",					/* name */
    FG_CYAN,						/* colour */

    NULL,						/* hdr */
    NULL,						/* body */
    &nautilus_nvlog					/* next */
};

/* unmarked, reserved ROM regions that should not be touched. */
rom_t *plat_rom_rsvd=&nautilus_nvenv;

/*----------------------------------------------------------------------*/
/* Private data and functions */

static unsigned rom_sectors[] = {
	0x000000,				/* 64 K */
	0x010000,				/* 64 K */
	0x020000,				/* 64 K */
	0x030000,				/* 64 K */
	0x040000,				/* 64 K */
	0x050000,				/* 64 K */
	0x060000,				/* 64 K */
	0x070000,				/* 64 K */
	0x080000,				/* 64 K */
	0x090000,				/* 64 K */
	0x0A0000,				/* 64 K */
	0x0B0000,				/* 64 K */
	0x0C0000,				/* 64 K */
	0x0D0000,				/* 64 K */
	0x0E0000,				/* 64 K */
	0x0F0000,				/* 32 K */
	0x0F8000,				/*  8 K */
	0x0FA000,				/*  8 K */
	0x0FC000,				/* 16 K */

	/* second ROM - be cautious here.  For some operations the 
	 * ROM takes sector addresses as args so we need to deduce the
	 * raw sector address instead of relative to the other ROM sometimes
	 */

	0x100000,				/* 64 K */
	0x110000,				/* 64 K */
	0x120000,				/* 64 K */
	0x130000,				/* 64 K */
	0x140000,				/* 64 K */
	0x150000,				/* 64 K */
	0x160000,				/* 64 K */
	0x170000,				/* 64 K */
	0x180000,				/* 64 K */
	0x190000,				/* 64 K */
	0x1A0000,				/* 64 K */
	0x1B0000,				/* 64 K */
	0x1C0000,				/* 64 K */
	0x1D0000,				/* 64 K */
	0x1E0000,				/* 64 K */
	0x1F0000,				/* 32 K */
	0x1F8000,				/*  8 K */
	0x1FA000,				/*  8 K */
	0x1FC000,				/* 16 K */
};


#define ROM_BASE 0xFFFC0000UL
#define ROM_NSECTS	( sizeof(rom_sectors) / sizeof(unsigned) )

#define RAW_SECTOR_MASK	0x0FFFFF	/* sector addrs relative to this ROM */
#define ROM_BASE_MASK	0x100000	/* for producing ROM base address */
#define BASEOF( x )	( (x) & ROM_BASE_MASK )
#define SECTOROF( x )	( (x) & ROM_SECTOR_MASK )


#define addrA 	0xAAA
#define addr5 	0x555

enum rom_states { ROM_CLOSED, ROM_OPEN };
static unsigned char wp_state = ROM_CLOSED;     /* write-protect on power-on */

static const unsigned long plat_romwindow = 256UL << 10;



/* a small bit of argument range validation */

static int validate_romargs( unsigned start, unsigned end )
{
    if ( (start < 0) || (start >= end) || (end >= plat_romsize) )
    {
        mobo_alertf( "Internal Error",
                    "I'm trying to ROM transactions beyond the address\r"
                    "range of the ROM [0x%04X->0x%04X].  Aborting [sorry]",
                    start, end );
        return FALSE;
    }
    return TRUE;			/* otherwise the args are okay */
}


/* Writes to set the GPIO bits for this address */
/* The GPIO bits for Nautilus are:
 * 6 - select/deselect upper 1M ROM
 * 0,1 - select which 256K page of selected ROM to use
 */

#define GPIO_MASK	0x43U		/* see comment above */

static void program_gpio( const unsigned offset )
{
    unsigned bits;

    /* Optimisation: cache the GPIO bits in the southbridge - everyone must
     * agree on using the southbridge to go via the cached copy
     */
    bits = ( offset / plat_romwindow ) & 0x3U;
    if ( offset >= (1<<20) )            /* use upper 1M ROM? */
	bits |= 0x40U;


    /* load in the GPIO cache */
    if ( ali_gpio_cached == FALSE )
	ali_load_gpio_cache();


    /* Update the ALI Southbridge GPIO bits where necessary using cached copies
     * and write back.  See Acer M1543C document, p150 */

    if ( (ali_gpio_dirn & GPIO_MASK) != GPIO_MASK )
    {
	ali_gpio_dirn |= GPIO_MASK;
	pcicfgwb( 0, PCI_ALI_PMU, 0, PMU_DCGPIO, ali_gpio_dirn );
    }

    if ( (ali_gpio_data & GPIO_MASK) != bits )
    {
	ali_gpio_data = (ali_gpio_data & ~GPIO_MASK) | bits;
	pcicfgwb( 0, PCI_ALI_PMU, 0, PMU_DOGPIO, ali_gpio_data );
    }
}



/* for communication with the flash firmware controller.
 * This routine doesn't program any GPIO bits, it now comes down to the 
 * caller to select the correct ROM to speak to.
 */

static void rom_ioctl( const unsigned offset, const unsigned val )
{
    if (offset > plat_romwindow )	/* DEBUG */
    {
	mobo_logf( LOG_WARN "ROM: Attempted access beyond ROM window!\n");
	return;
    }

    return outmemb( ROM_BASE + offset, val );	/* perform PCI-mem write */
}


static unsigned is_sector_protected( const unsigned secno )
{
    unsigned rval;
    unsigned saddr = rom_sectors[secno] + 4;	  /* 4->byte width */

    /* first we need to select the right ROM to program */
    program_gpio( BASEOF( saddr ) );

    rom_ioctl( addrA, 0xAA );
    rom_ioctl( addr5, 0x55 );		/* into autoselect mode */
    rom_ioctl( addrA, 0x90 );		/* autoselect command */

    rval = plat_inromb( saddr );	/* read sector, nb gpio setup is done */

    program_gpio( BASEOF( saddr ) );
    rom_ioctl( 0, 0xF0 );		/* exit autoselect mode */

    return rval;
}



/* erase an individual sector, passed as an index into the array above */
/* this is an internal function, writes must already be enabled to ROM */
/* For reference, see the AMD Am29F800B data sheet */

static void erase_sector( const unsigned secno )
{
    unsigned saddr = rom_sectors[secno];

    program_gpio( BASEOF( saddr ) );		/* select the correct ROM */

    rom_ioctl( addrA, 0xAA );
    rom_ioctl( addr5, 0x55 );
    rom_ioctl( addrA, 0x80 );
    rom_ioctl( addrA, 0xAA );
    rom_ioctl( addr5, 0x55 );

    program_gpio( saddr );			/* correct window in the ROM */
    rom_ioctl( saddr % plat_romwindow, 0x30 );
}


#define ROM_TIMEOUT     2000UL		/* in msec, probably far too much */

static DBM_STATUS wait_rom_ready( const unsigned addr,
				  const unsigned char rdysign )
{
    unsigned long start, current, elapsed, end;
    unsigned char c;

    TRACE( "Running\n" );

    /* timeout expiry calculation */
    end = (ROM_TIMEOUT * 1000000000UL) / primary_impure->CYCLE_CNT;
    start = rpcc();

    while( (c=plat_inromb( addr )) != rdysign) {
        current = rpcc();
        if ( current <= start )         current += 1UL<<32;     /* wraparound */
        elapsed = current - start;
        if ( elapsed > end ) {
            mobo_logf( LOG_CRIT "ROM: wait_rom_ready timeout after %ld ticks, "
                                "want 0x%02X got 0x%02X at address 0x%X\n",
                                elapsed, rdysign, c, addr );
            return STATUS_FAILURE;
        }
    }

    return STATUS_SUCCESS;
}


static void open_rom( void )		/* disable write-protection to ROM */
{
    unsigned bcsc;

    /* enable writes to both ROMs simulatneously via BCSC bit 6 in M1543C */
    bcsc = pcicfgrb( 0, PCI_ALI_ISA, 0, ISA_BCSC );
    bcsc |= 1 << 6;
    pcicfgwb( 0, PCI_ALI_ISA, 0, ISA_BCSC, bcsc );
}

static void close_rom( void ) 		/* enable write_protection to ROM */
{
    unsigned bcsc;

    /* disable writes to the ROM via BCSC register bit 6 in M1543C */
    bcsc = pcicfgrb( 0, PCI_ALI_ISA, 0, ISA_BCSC );
    bcsc &= ~(1 << 6);
    pcicfgwb( 0, PCI_ALI_ISA, 0, ISA_BCSC, bcsc );
}


/* perform the transaction to write to the ROM, without erase. */

DBM_STATUS plat_outromb( const unsigned addr, const unsigned char data )
{
    unsigned char rom_state;
    DBM_STATUS sval;

    /* sanity check: is the address within my ROM size? */
    if (addr > plat_romsize )                           return;

    /* fiddly detail - if the ROM is not open at present, we need to do it */
    rom_state = wp_state;
    if ( rom_state == ROM_CLOSED )      open_rom();

    program_gpio( BASEOF( addr ) );		/* select correct ROM */
    rom_ioctl( addrA, 0xAA );
    rom_ioctl( addr5, 0x55 );
    rom_ioctl( addrA, 0xA0 );

    program_gpio( addr );			/* window within the ROM */
    rom_ioctl( addr % plat_romwindow, data );

    sval = wait_rom_ready( addr, data );
    if ( sval != STATUS_SUCCESS )
    {
        mobo_logf( LOG_CRIT "ROM: Write operation failed at address 0x%X\n",
                addr );
    }

    if ( rom_state == ROM_CLOSED )
	close_rom();
    return sval;
}



/*----------------------------------------------------------------------*/
/* Platform ROM access interface */

/* ROM sector handling stuff:
 * find upper and lower sector bounds of an address 
 */

unsigned plat_romlower( unsigned addr )
{
    /* find the first address in this sector */
    unsigned sec, rval=0;

    for ( sec = 0; sec < ROM_NSECTS; sec++ )
    {
	if ( rom_sectors[sec] <= addr )		rval = rom_sectors[sec];
	else break;
    }
    return rval;
}

unsigned plat_romupper( unsigned addr )
{
    /* find the last address in this sector, */
    /* ie 1 below first address in next sector */
    unsigned sec;

    for ( sec = 0; sec < ROM_NSECTS; sec++ )
    {
	if ( rom_sectors[sec] > addr )
		return rom_sectors[sec] - 1;
    }
    return plat_romsize - 1;		/* for last sector (or way out) */
}


unsigned plat_inroml( const unsigned offset )
{
    unsigned waddr;

    /* is the address within my ROM size?  Is it naturally aligned? */
    if (offset > plat_romsize || offset & 0x3 )		return 0xFFFFFFFFU;

    program_gpio( offset );			/* setup overlay window */
    waddr = offset % plat_romwindow;		/* truncate ptr to overlay  */
    return inmeml( ROM_BASE + waddr );		/* perform PCI-mem read */
}


unsigned char plat_inromb( const unsigned offset )
{
    unsigned waddr;

    /* is the address within my ROM size? */
    if (offset > plat_romsize )			return 0xFF;

    program_gpio( offset );			/* setup overlay window */
    waddr = offset % plat_romwindow;		/* truncate ptr to overlay  */
    return inmemb( ROM_BASE + waddr );		/* perform PCI-mem read */
}


DBM_STATUS plat_romerase( unsigned start, unsigned end, 
			  Callback_t CB, unsigned ncb )
{
    DBM_STATUS sval;
    unsigned ssec, esec, sec;
    unsigned lastCB=0;
    unsigned last;

    /* argument validation */
    if ( ! validate_romargs( start, end ) )	return STATUS_FAILURE;


    /* calculate the range to erase.  Hopefully erasing a firmware image 
     * doesn't also wipe out parts of other images */

    /* [caller should also do this and validate there is no overlap] */
    start = plat_romlower( start ), end = plat_romupper( end );

    /* look up sector numbers for the address range */
    for ( ssec=0; ssec < ROM_NSECTS; ssec++ )
	if ( rom_sectors[ssec] > start )	break;
    ssec--;
    for ( esec=0; esec < ROM_NSECTS; esec++ )
	if ( rom_sectors[esec] > end ) 		break;
    esec--;

    open_rom();				/* disable ROM's write-protection */

    /* validate whether each sector is protected or not */
    for ( sec = ssec; sec <= esec; sec++ )
    {
	if ( is_sector_protected( sec ) ) {
	    mobo_alertf( "ROM Write Protection",
		"At least one sector (%d, 0x%06X) of the chosen range\r"
		"is write-protected.  Aborting [sorry]",
		sec, rom_sectors[sec] );
	    close_rom();
	    return STATUS_FAILURE;
	}
    }

    /* perform the wipe operation by queuing requests to the chip */
    for ( sec = ssec; sec <= esec; sec++ )
    {
	if ( CB != NULL )				/* Callbacks wanted */
	{
	    /* a little complex here because we must consider the possibility
	     * that one sector op could cover many callback blocks */

	    if ( lastCB==0 )			/* fill in first block */
	    {
		CB();
		lastCB = rom_sectors[ssec];
	    }

	    /* work out how many screen blocks this ROM sector entails */
	    /* NB must cater for more-than and less-than one cases */
	    last = plat_romupper( rom_sectors[sec] );

	    while ( (last - lastCB) >= ncb ) 
	    {
		CB();
		lastCB += ncb;
	    }
	}

	erase_sector( sec );

	/* 0xFF=erased state */
        sval = wait_rom_ready( rom_sectors[sec], 0xFF );
	if (sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_CRIT
			"ROM: didn't erase to blank state.\n" );
	    return sval;
	}
    }

    /* re-enable ROM write protection and quit */
    close_rom();

    return STATUS_SUCCESS;
}


DBM_STATUS plat_romwrite( unsigned start, const unsigned char *img, unsigned nb,
		   Callback_t CB, unsigned ncb )
{
    unsigned i, addr;
    unsigned lastCB=0;
    DBM_STATUS sval;

    /* argument validation */
    if ( ! validate_romargs( start, start + nb - 1) )	return STATUS_FAILURE;

    open_rom();				/* prepare to write */

    for ( i=0, addr=start; i < nb; i++, addr++ )
    { 
        if ( CB != NULL )                               /* Callbacks wanted */
        {
            if ( lastCB==0 )                    /* fill in first block */
            {
                CB();
                lastCB = addr;
            }

	    if ( (addr - lastCB) >= ncb ) 
            {
                CB();
                lastCB += ncb;
            }
        }

	sval = plat_outromb( addr, img[i] );
	if ( sval == STATUS_FAILURE )
	{
            mobo_logf( LOG_CRIT
                        "ROM: Write to ROM at 0x%X produced corrupted data\n",
                        addr );
	    mobo_alertf( "Write failed!",
		"The ROM is not responding correctly to write operations.\r"
		"The reflash operation cannot complete!" );
	    break;
        }
    }

    close_rom();
    return sval;
}

