/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* System-related configuration details for the API Swordfish platform */
/* Begun by Stig Telfer, Alpha Processor Inc, 29 June 1999 */

#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"
#include "rom.h"

#include "specifics.h"			/* motherboard specific definitions */
#include "platform.h"			/* interface for this module */
#include "northbridge.h"


/* The Swordfish ROM is an AM29F016B, which is a 2Mbyte, 8-bit wide ROM
 * on the TIG bus, which hangs off the Tsunami C-Chip */


/*----------------------------------------------------------------------*/
/* Reading from a ROM appears to be a highly platform-dependent thing
 */


const unsigned long plat_romsize = 2UL << 20;	/* ROM is 2Mbyte total */

#ifdef CONFIG_SWORDFISH
static rom_t sword_nvenv = {

    0,                                                  /* hstart */
    0x5F800U, 0x800U,					/* rstart, rsize */
    0x50000U, 0x10000U,					/* astart, asize */

    ROM_IMGVALID | ROM_PLATRSVD,			/* flags */
    FW_NVENV,						/* fwid */

    "System NVRAM",					/* name */
    FG_CYAN,                                            /* colour */

    NULL,                                               /* hdr */
    NULL,						/* body */
    NULL						/* next */
};

static rom_t sword_nvlog = {

    0,							/* hstart */
    0x1F0000U, 0x10000U,				/* rstart, rsize */
    0x1F0000U, 0x10000U,				/* astart, asize */

    ROM_IMGVALID | ROM_PLATRSVD,			/* flags */
    FW_NVLOG,                                           /* fwid */

    "Diags log",					/* name */
    FG_CYAN,                                            /* colour */

    NULL,                                               /* hdr */
    NULL,						/* body */
    &sword_nvenv					/* next */
};


/* unmarked, reserved ROM regions that should not be touched. */
rom_t *plat_rom_rsvd=&sword_nvlog;               
#endif	/* CONFIG_SWORDFISH */

#ifdef CONFIG_SHARK
static rom_t shark_nvenv = {

    0,                                                  /* hstart */
    0x4F800U, 0x800U,					/* rstart, rsize */
    0x40000U, 0x10000U,					/* astart, asize */

    ROM_IMGVALID | ROM_PLATRSVD,			/* flags */
    FW_NVENV,						/* fwid */

    "System NVRAM",					/* name */
    FG_CYAN,                                            /* colour */

    NULL,                                               /* hdr */
    NULL,						/* body */
    NULL						/* next */
};

/* unmarked, reserved ROM regions that should not be touched. */
rom_t *plat_rom_rsvd=&shark_nvenv;               
#endif	/* CONFIG_SHARK */


/*----------------------------------------------------------------------*/
/*
 * Private ROM access functions.
 */
#define addr555 0x555
#define addr2AA 0x2AA

enum rom_states { ROM_CLOSED, ROM_OPEN };

static unsigned char wp_state = ROM_CLOSED;	/* write-protect on power-on */
static const unsigned sector_size = 64 << 10;	/* ROM is in 64K sectors */

/* These TIGBus addresses are magic numbers drawn from the debug monitor code
 * for the DP264 */

static void open_rom( void )
{
    TRACE( "Running\n" );
    tigwb( TIG_ROMENA, 1 );
    wp_state = ROM_OPEN;
}

static void close_rom( void )
{
    TRACE( "Running\n" );
    tigwb( TIG_ROMENA, 0 );
    wp_state = ROM_CLOSED;
}


#define ROM_TIMEOUT	2000UL		/* in msec, probably far too much */

static DBM_STATUS wait_rom_ready( const unsigned addr,
				  const unsigned char rdysign )
{
    unsigned long start, current, elapsed, end;
    unsigned char c;

    TRACE( "Running\n" );

    /* timeout expiry calculation */
    end = (ROM_TIMEOUT * 1000000000UL) / primary_impure->CYCLE_CNT;
    start = rpcc();

    while( (c=plat_inromb( addr )) != rdysign) {
	current = rpcc();
	if ( current <= start )		current += 1UL<<32;	/* wraparound */
	elapsed = current - start;
	if ( elapsed > end ) {
	    mobo_logf( LOG_CRIT "ROM: wait_rom_ready timeout after %ld ticks, "
				"want 0x%02X got 0x%02X at address 0x%X\n",
				elapsed, rdysign, c, addr );
	    return STATUS_FAILURE;
	}
    }

    return STATUS_SUCCESS;
}



/* [for communication with the flash firmware controller] */

static void rom_ioctl( const unsigned offset, const unsigned val )
{
    /* is the address within my ROM size? */
    if (offset > plat_romsize )                         return;
    tigwb( offset, val );		/* put the data onto the TIG */
}


static unsigned is_sector_protected( unsigned secno )
{
    unsigned rval;
    unsigned saddr = (secno * sector_size) | 2;	/* add the 2, as per spec */

    TRACE( "Running\n" );

    rom_ioctl( addr555, 0xAA );
    rom_ioctl( addr2AA, 0x55 );		/* into autoselect mode */
    rom_ioctl( addr555, 0x90 );		/* autoselect command */

    rval = plat_inromb( saddr );

    /* issue a reset command to exit autoselect mode */
    rom_ioctl( 0, 0xF0 );

    return rval;
}


/* writes must already be enabled to ROM */
/* For reference, see the AMD Am29F800B data sheet */
static void erase_sector( unsigned secno )
{
    unsigned saddr = secno * sector_size;

    TRACE( "Running\n" );

    rom_ioctl( addr555, 0xAA );
    rom_ioctl( addr2AA, 0x55 );
    rom_ioctl( addr555, 0x80 );
    rom_ioctl( addr555, 0xAA );
    rom_ioctl( addr2AA, 0x55 );
    rom_ioctl( saddr, 0x30 );
}




static int validate_romargs( int start, int end )
{
    TRACE( "Running\n" );

    /* argument validation */
    if ( (start < 0) || (start >= end) || (end >= plat_romsize) )
    {
        mobo_alertf( "Internal Error",
                    "I'm trying to do ROM transactions beyond the address\r"
                    "range of the ROM [0x%04X->0x%04X].  Aborting [sorry]",
                    start, end );
        return FALSE;
    }
    return TRUE;                        /* otherwise the args are okay */
}


unsigned plat_romlower( unsigned addr )
{
    if ( addr >= plat_romsize )		addr = plat_romsize - 1;
    return (addr & ~0xFFFF);		/* strip down to 64K boundary */
}

unsigned plat_romupper( unsigned addr )
{
    if ( addr >= plat_romsize )		addr = plat_romsize - 1;
    return (addr | 0xFFFF );		/* round up to 1 below next 64K */
}


unsigned plat_inroml( const unsigned offset )
{
    unsigned val;

    /* is the address within my ROM size?  Is it naturally aligned? */
    if (offset > plat_romsize || offset & 0x3 )		return 0xFFFFFFFF;

    val = plat_inromb( offset ) |
	  ( plat_inromb( offset+1 ) << 8) |
	  ( plat_inromb( offset+2 ) << 16)|
	  ( plat_inromb( offset+3 ) << 24);

    return val;
}


DBM_STATUS plat_outroml( const unsigned offset, const unsigned data )
{
    unsigned char rom_state;

    /* is the address within my ROM size?  Is it naturally aligned? */
    if (offset > plat_romsize || offset & 0x3 )		return STATUS_FAILURE;

    /* fiddly detail - if the ROM is not open at present, we need to do it */
    rom_state = wp_state;
    if ( rom_state == ROM_CLOSED )	open_rom();

    if( plat_outromb( offset, data & 0xFF ) == STATUS_FAILURE 	||
	plat_outromb( offset+1, (data >>  8) & 0xFF ) == STATUS_FAILURE ||
	plat_outromb( offset+2, (data >> 16) & 0xFF ) == STATUS_FAILURE ||
	plat_outromb( offset+3, (data >> 24) & 0xFF ) == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "ROM: Write failed at address 0x%X\n", offset );
	if ( rom_state == ROM_CLOSED )
	    close_rom();
	return STATUS_FAILURE;
    }

    if ( rom_state == ROM_CLOSED )
	close_rom();
    return STATUS_SUCCESS;
}


unsigned char plat_inromb( const unsigned offset )
{
    /* is the address within my ROM size? */
    if (offset > plat_romsize )				return 0xFF;

    return tigrb( offset );
}


/* perform the transaction to write a byte to the ROM, without erase. */

DBM_STATUS plat_outromb( const unsigned addr, const unsigned char data )
{
    unsigned char rom_state;
    DBM_STATUS sval;

    /* is the address within my ROM size? */
    if (addr >= plat_romsize )				return STATUS_FAILURE;

    /* fiddly detail - if the ROM is not open at present, we need to do it */
    rom_state = wp_state;
    if ( rom_state == ROM_CLOSED )      open_rom();

    rom_ioctl( addr555, 0xAA );
    rom_ioctl( addr2AA, 0x55 );
    rom_ioctl( addr555, 0xA0 );
    rom_ioctl( addr, data );

    sval = wait_rom_ready( addr, data );
    if ( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT "ROM: Write operation failed at address 0x%X\n",
		addr );
    }

    if ( rom_state == ROM_CLOSED )
	close_rom();
    return sval;
}






DBM_STATUS plat_romerase( unsigned start, unsigned end, Callback_t CB, unsigned ncb )
{
    unsigned startsec, endsec, sec;
    unsigned progress, lastCB=0;

    TRACE( "Running\n" );

    /* argument validation */
    if ( ! validate_romargs( start, end ) )     return STATUS_FAILURE;

    /* calculate how many sectors to erase.  Hopefully, the ROM will be
     * organised in such a way that firmware images start on a sector
     * boundary.  If not, warn about this. */

    startsec = start / sector_size;
    if ( start % sector_size )
            mobo_alertf("Firmware image oddity!",
                        "The address range being erased isn't on a ROM\r"
                        "sector boundary.  Erasure will start at %d KB",
                        startsec * sector_size >> 10 );

    endsec = end / sector_size;
    if ( end % sector_size ) 	end++; 		/* round up */

    open_rom( );				/* not sure if required? */

    /* validate whether each sector is protected or not */
    for ( sec = startsec; sec <= endsec; sec++ )
    {
        if ( is_sector_protected( sec ) ) {
            mobo_alertf( "ROM Write Protection",
                        "At least one sector of the chosen region of ROM\r"
                        "is write-protected.  Aborting [sorry]" );
            close_rom();
            return STATUS_FAILURE;
        }
    }

    /* perform the wipe operation by queuing requests to the chip */
    for ( sec = startsec; sec <= endsec; sec++ )
    {
        erase_sector( sec );

        /* 0xFF=erased state */
        if ( wait_rom_ready( sec * sector_size, 0xFF ) != STATUS_SUCCESS )
	{
	    mobo_alertf( "ROM not responding",
		"The ROM is not responding to erase commands at address 0x%X",
		sec * sector_size );
	    mobo_logf( LOG_CRIT "ROM: Erase command failed at address 0x%X\n",
		sec * sector_size );
	    close_rom();
	    return STATUS_FAILURE;
	}

	if ( CB != NULL )
	{
	    if ( lastCB==0 ) 				/* fill in first blk */
	    {
		CB();
		lastCB = startsec * sector_size;
	    }

	    for ( progress = sec * sector_size;
		  progress - lastCB > ncb; lastCB += ncb )
			CB();
	}
    }

    /* re-enable ROM write protection and quit */
    close_rom();

    return STATUS_SUCCESS;
}





DBM_STATUS plat_romwrite( unsigned start, const unsigned char *img, unsigned nb,
                   Callback_t CB, unsigned ncb )
{
    unsigned i, addr;
    DBM_STATUS sval = STATUS_SUCCESS;

    TRACE( "Running\n" );

    /* argument validation */
    if ( ! validate_romargs( start, start + nb - 1 ) )
	return STATUS_FAILURE;

    open_rom();                         /* prepare to write */

    for ( i=0, addr=start; i < nb; i++, addr++ )
    {
        sval = plat_outromb( addr, img[i] );
	if ( sval == STATUS_FAILURE )
	{
	    mobo_logf( LOG_CRIT 
			"ROM: Write to ROM at 0x%X produced corrupted data\n",
			addr );
            mobo_alertf( "Write failed!",
                "The ROM is not responding correctly to write operations.\r"                    "The reflash operation cannot complete!" );
	    break;
	}

	if ( CB != NULL )
	    if ( i % ncb == 0 )		CB();
    }

    close_rom();
    return sval;
}


