/* tFIFO.h - define interface to FIFO routines
 * Copyright  1997-1999 by Object Software Inc. - All Rights Reserved
 *
 * The authors hereby grant permission to use, copy, modify, distribute,
 * and license this software and its documentation for any purpose, provided
 * that existing copyright notices are retained in all copies and that this
 * notice is included verbatim in any distributions. No written agreement,
 * license, or royalty fee is required for any of the authorized uses.
 */

/* tFIFO inplements a first-in, first-out circular buffer of bytes for I/O buffering. */

#ifndef	TFIFO_H
#define	TFIFO_H

#ifdef __cplusplus
extern "C" {
#endif

/* Declare the FIFO data type. */
typedef struct sFIFO
{
	char *Buffer;				// buffer where bytes are stored.
	unsigned BufferSize;		// capacity of the buffer. MUST BE A POWER OF TWO!
	unsigned InPtr;				// The current write position
	unsigned OutPtr;			// The current read position
} tFIFO;

/* Read up to (Max) bytes from FIFO, returning the actual count read. */
int ReadFIFO (tFIFO *FIFO, int Max, char *Dest);

/* Write up to (Max) bytes to FIFO, returning the actual count written. */
int WriteFIFO (tFIFO *FIFO, int Max, char *Source);

/* Find out how many bytes are stored in the FIFO. */
unsigned FIFOCount (tFIFO *FIFO);

/* Returns non-zero if the FIFO is empty. */
int FIFOEmpty (tFIFO *FIFO);

/* Inline version of FIFOEmpty. */
static int InlineFIFOEmpty (tFIFO *FIFO)
{
	return FIFO->InPtr == FIFO->OutPtr;
}

/* Returns non-zero if the FIFO is full. */
int FIFOFull (tFIFO *FIFO);

/* Inline version of FIFOFull. */
static inline int InlineFIFOFull (tFIFO *FIFO)
{
	return FIFO->OutPtr == ((FIFO->InPtr + 1) & (FIFO->BufferSize - 1));
}

/* Inline routine to write a single character to a FIFO.
 * Returns non-zero if successful, 0 if FIFO full.
 */
static inline int FIFOputc (tFIFO *FIFO, char c)
{
	FIFO->Buffer [FIFO->InPtr] = c;
	if (InlineFIFOFull (FIFO))
		return 0;
	if (FIFO->InPtr == FIFO->BufferSize - 1)
		FIFO->InPtr = 0;
	else FIFO->InPtr++;
	return ~0;
}

/* Inline routine to read a single character from a FIFO.
 * Returns zero if the FIFO is empty.
 */
static inline int FIFOgetc (tFIFO *FIFO, char *Dest)
{
	if (InlineFIFOEmpty (FIFO))
		return 0;
	*Dest = FIFO->Buffer [FIFO->OutPtr];
	if (FIFO->OutPtr == FIFO->BufferSize - 1)
		FIFO->OutPtr = 0;
	else FIFO->OutPtr++;
	return ~0;
}

#ifdef __cplusplus
}
#endif

#endif
