; HP 9895A floppy disk firmware
; Partially reverse-engineered by Eric Smith <spacewar@gmail.com>
; and Craig Ruff

; Cross-assembles with Macro Assembler AS:
;   http://john.ccac.rwth-aachen.de:8000/as/

        page 0
        cpu z80undoc

        include macros.inc
        include phi.inc
        include io.inc
        include ram.inc

        equ ID_BYTE_1, 00h
        equ ID_BYTE_2, 81h

; ***************************************************************
; * ROM ADDRESS SPACE                                           *
; *                                                             *
; * Note: unused parts of ROM filled with 76h (halt) bytes      *
; ***************************************************************
        section rom

start equ 0000h
size  equ 2000h

	org	start
        jp	do_reset                ; rst 00h location

wait_for_cmd_jv:
        jp	do_wait_for_cmd

;x0006:
        db      12h, 80h
        ;ld	(de), a
	;add	a, b

reset_drive macro
        rst     08h
        endm
reset_drive_jv:                         ; rst 08h location
        jp	do_reset_drive

check_for_command_byte_jv:
        jp	do_check_for_command_byte

;x000e:
        db      06h, 26h
	;ld	b, 26h

seek    macro
        rst     10h
        endm
        jp	do_seek                 ; rst 10h location

HP_300_Clear_jv:
        jp	do_HP_300_Clear

;x0016:
        db      16h, 54h
	;ld	d, 54h

wait_for_sector macro
        rst     18h
        endm
        jp	do_wait_for_sector      ; rst 18h location

step_drive_jv:
        jp	do_step_drive

;x001e:
        db      4dh, 60h
	;ld	c, l
	;ld	h, b

load_head macro
        rst   20h
        endm
        jp	do_load_head            ; rst 20h location

determine_unit_status_jv:
        jp	do_determine_unit_status

	fillto	0028h, 076h

set_in_use_leds macro
        rst     28h
        endm
show_in_use_leds_jv:                    ; rst 28h location
        jp	do_show_in_use_leds

read_sector_jv:
        jp	do_read_sector

	fillto	0030h, 076h

; turns off the in use LED
; unloads heads
deassert_in_use macro
        rst     30h
        endm
        jp	do_deassert_in_use      ; rst 30h location

write_sector_jv:
        jp	write_sector

	fillto	0038h, 076h

;******************************************************************
; 38h: IM1 vector is entered when a PHI I/O is terminated
im1_jv: ex	af, af'                 ; Save current flags
	di
        in	a, (phi_status)         ; Get the D0D1 bits
	exx
	ld	hl, terminate_listen_tranfer
	ex	(sp), hl
	exx
	push	af
	ex	af, af'                 ; Restore saved flags
	pop	af                      ; Return with phi_status value in A
	reti                            ; Jumps to terminate_listen_tranfer

	fillto	0057h, 076h

x0057:  jp	deselect_drives         ; not reached??
	jp	prep_drive_regs         ; not reached??

fmt_hp_track:
        jp	do_hp_fmt_track

fmt_ibm_track_jv:
        jp	fmt_ibm_track

verify_cylinder_jv:
        jp	verify_cylinder

;******************************************************************
; 66h: NMI vector is entered for other PHI interrupts
nmi_jv:
        jp	do_phi_int

	fillto	006ah, 076h

seek_to_cylinder_jv:
        jp	do_seek_to_cylinder

send_status_or_address_jv:
        jp	do_send_status_or_address

done_holdoff:
        ld	a, DSJ_holdoff
	jr	set_DSJ

; done_normal - normal command completion and cleanup
done_normal:
        xor	a                               ; A = DSJ_normal

;******************************************************************
; set_DSJ - non-normal command completion and cleanup
;
; on entry:
;       A: the DSJ value to set

set_DSJ:
        ld	(DSJ), a
	ld	sp, ramend                      ; Reset SP for next command
	xor	a                               ; Clear PHI D0, D1 bits
	out	(phi_status), a
	ld	(parity_check_state), a         ; disable HP-IB parity checking
	ld	a, FIFO_ROOM_m | FIFO_BYTE_m | DEV_CLR_m
	out	(phi_int_mask), a
	in	a, (phi_control)	        ; clear parity freeze
	res	PRTY_FRZ, a
	out	(phi_control), a

do_device_clear:
        xor	a
	ld	(S1), a                         ; S1_NORMAL
	ld	(S1_unit), a                    ; Unit 0
	ld	(head_unload_set), a            ; clear head_unload_set
	call	prep_for_next_cmd
	call	set_parity_check_state
	xor	a
	out	(cntl), a                       ; turn off write mode, etc
	out	(phi_status), a                 ; D0D1=00
	out	(phi_id1), a                    ; set Identify bytes (ID_BYTE_1 = 00h)
	ld	a, ID_BYTE_2
	out	(phi_id2), a
	ld	a, PRTY_ERR_m | DATA_FRZ_m      ; D0D1=01, clear output data freeze
	out	(phi_status), a
	ld	a, 0ffh                         ; enable all interrupt conditions
	out	(phi_int_cond), a
	ld	(iy + YB(tmp_unit)), NUNITS - 1 ; initialize the DriveData structures

.drive_init_loop:
        ld	a, (tmp_unit)
	cp	-1
	jr	z, exit_drive_init_loop
	dec	(iy + YB(tmp_unit))
	ld	b, a
	call	get_drive_data_ptr_in_ix
	xor	a
	out	(drv), a                        ; reset drv register (head 0, not in use)
	ld	(ix + DriveData.drv), a         ; Keep a copy

	inc	b                               ; calculate the drive's select bit #(3 - unit)
	ld	a, DRV0_m << 1
.loop:	rrca
	djnz	.loop

	ld	(ix + DriveData.xv), a          ; save it for later use (clearing the other bits)
	out	(xv), a                         ; select the drive too
	call	get_drive_SS                    ; get the drive's current status
	jr	z, .no_disc_or_fs
	ld	a, (DSJ)
	cp	DSJ_holdoff
	jr	z, .first_status                ; in holdoff
	ld	a, (ix + DriveData.stat2)
	and	DISC_TYPE_m
	ld	d, a                            ; stat2 (not its usual register assignment)
	ld	a, (ix + DriveData.stat2_ds)
	and	STAT2_W_m | STAT2_F_m | SS_m
	ld	e, a                            ; stat2_ds (not its usual register assignment)
	and	SS_m                            ; isolate stat2 SS, clear the other bits - seems unused
	jr	.set_status

.no_disc_or_fs:
        jr	nc, .no_disc
.first_status:
        ld	d, 0                            ; clear stat2
	ld	e, STAT2_F_m
	jr	.set_status

.no_disc:
        ld	e, a
	ld	d, 0                            ; clear stat2
.set_status:
        ld	(ix + DriveData.stat2), d
	ld	(ix + DriveData.stat2_ds), e
	reset_drive
	jr	z, .skip_c
	ld	(ix + DriveData.stat2_ds), STAT2_C_m    ; seek check
.skip_c:
        call	deselect_drives
	jr	.drive_init_loop

exit_drive_init_loop:
        in	a, (phi_control)
	and	PRTY_FRZ_m                      ; save the parity freeze state
	or	EIGHT_BIT_m | RSPD_PP_m | INIT_FIFO_m
	ld	b, a                            ; set initial PHI control state
	ld	a, POLL_HLDF
	out	(phi_status), a
	ld	a, b
	out	(phi_control), a

	in	a, (switches)
	and	ADDRESS_PHI_m                   ; read the HP-IB address from the switches
	or	ONL_m                           ; set the HP-IB online state

        ; Possible bug here? D0=1 from previous write to phi_status above,
        ; but that will cause the CRCE bit to be enabled for this write to
        ; phi_address. Will that enable the PHI CRC SAD 11h support even
        ; though the service manual says they are supposed to be ignored?
	out	(phi_address), a
	im	1                               ; Interrupt mode 1 vectors INT to 038h
	ld	a, (DSJ)
	cp	DSJ_holdoff
	jr	nz, do_wait_for_cmd             ; not holdoff
	jr	show_idle_state

;******************************************************************
; do_wait_for_cmd - the main idle loop

do_wait_for_cmd:
        call	prep_for_next_cmd
	ld	a, (DSJ)                        ; form the waiting + DSJ state pattern
	rlca                                    ; put DSJ value into proper position
	or	IU_WAIT_DSJ
	set_in_use_leds

cmd_wait_loop:
        ld	b, 0
;x0137:
.inner:	call	poll_input_or_DCL
	jr	nz, .exitloop
	djnz	.inner                          ; not 256 interations yet
	ld	d, 16                           ; approx 1/16 s elapsed?
	call	head_unload_timer_check
	jr	cmd_wait_loop

.exitloop:
        call	read_Amigo_cmd
	jr	do_wait_for_cmd

;******************************************************************
; show_idle_state - entered after a device clear has executed
;
; Lighting the DONE LED shows either self test has stopped or that
; the 9895A is not under active control of an HP-IB controller.

show_idle_state:
        ld	a, (leds_shdw)
	set	DONE, a
	cpl                                     ; (Complement for LED OC drivers)
	out	(leds), a                       ; Show it
	jr	cmd_wait_loop

;******************************************************************
; read_Amigo_cmd - read an Amigo command and process it

read_Amigo_cmd:
        call	do_PPD
	ld	a, ~IU_RECV                     ; OC LED drivers need complemented data
	out	(leds), a

.wait_for_sad:
        in	a, (phi_fifo)                   ; Read the HP-IB data byte
	ld	c, a
	in	a, (phi_status)
	bit	SAD_D1, a                       ; Was D1 set (maybe rcvd a secondary address)?
	jr	nz, .handle_sad
	call	wait_for_input_or_DCL
	jr	.wait_for_sad

.handle_sad:
        and	D0D1_m                          ; isolate D0D1 bits
	cp	SAD_D1_m                        ; secondary address?
	jp	nz, abort_IO_program_error      ; no, EOI
	ld	a, c                            ; handle secondary address
	cp	DSJ_SAD
	jp	z, cmd_DSJ                      ; yes, handle DSJ command
	call	check_for_parity_error
	ld	a, c		                ; A = secondary address
	cp	HP300_CLEAR_SAD
	jp	z, do_HP_300_Clear              ; yes, HP-300 clear (listener)

	and	SAD_m
	ld	hl, SAD_dispatch_table
	ld	de, 3                           ; ? sets D = 0, E gets overwritten later
	ld	b, 9                            ; SAD_dispatch_table length
.loop:	cp	(hl)
	jr	z, .matched
	add	hl, de
	djnz	.loop
	jp	abort_IO_program_error          ; no match found

.matched:
        inc	hl                              ; HL points to listen SAD entry
	bit	TLK, c                          ; addressed to talk?
	jr	z, .skip
	inc	hl                              ; HL points to talk SAD entry
.skip:	ld	e, (hl)                         ; find the jump table entry's address
	ld	hl, SAD_jmptbl
	add	hl, de

;******************************************************************
; dispatch_cmd - execute a command handler
;
; on entry:
;       HL: pointer to the handler's address

dispatch_cmd:
        ld	e, (hl)
	inc	hl
	ld	h, (hl)
	ld	l, e
	
	ld	de, ignore_cmd  ; in case (HL) returns?
	push	de
	jp	(hl)	        ; call subroutine pointed to by HL

ignore_cmd:
	call	poll_input_or_DCL
	ret	nz
	call	do_PPE
	ret

;******************************************************************
; wait_for_input_or_DCL - wait until there are input bytes or a DCL
;       If we've been addressed to talk or listen in this state do nothing until
;       we time out as we are supposed to be reading command data.
wait_for_input_or_DCL:
        ld	bc, 0
	ld	d, 31
.loop:
        call	poll_input_or_DCL
	ret	nz
	in	a, (phi_status)
	and	TLK_IDF_m | LTN_m               ; addressed to talk or listen?
	jp	z, abort_IO_program_error       ; no
	djnz	.loop
	dec	c
	jr	nz, .loop
	dec	d
	jr	nz, .loop
	jp	abort_IO_program_error          ; timed out

;******************************************************************
; secondary address (SAD) dispatch data tables
;   first byte is secondary address
;   second and third bytes are byte indexes into jump table at SAD_jmptbl
;   second byte = listen, third byte = talk
SAD_dispatch_table:
        db	08h, 04h, 00h
	db	09h, 04h, 00h
	db	0ah, 04h, 00h
	db	0bh, 04h, 00h
	db	0ch, 04h, 00h
	db	0fh, 10h, 00h
	db	11h, 06h, 02h
	db	1eh, 08h, 0ah
	db	1fh, 0ch, 0eh
	
SAD_jmptbl:
        dw	abort_IO_program_error	; invalid or send status or address??
	dw	terminate_talk	        ; talker   11: HP-IB CRC
	dw	do_unit_cmds	        ; listener 08, 09, 0a, 0b, 0c: commands
	dw	ignore_cmd	        ; listener 11: HP-IB CRC
	dw	write_loopback	        ; listener 1e
	dw	read_loopback	        ; talker   1e
	dw	initiate_self_test	; listener 1f
	dw	read_self_test	        ; talker   1f
	dw	download	        ; listener 0f


;******************************************************************
; do_unit_cmds - handle reading command data bytes for drive related commands.
;                listen with secondary address 08, 09, 0a, 0b, 0c (via tables)
;                and 10 (HP-300 Clear) explicitly
; on entry:
;       C: the SAD byte that sent us here
do_unit_cmds:
        ld	hl, cmd_count
	push	bc
	push	hl
	ld	e, 0                    ; E: count of bytes read
	inc	hl                      ; buffer follows cmd_count in memory
.loop:	call	wait_for_input_or_DCL
	ld	c, phi_fifo
	ini                             ; read a byte from the controller
        in	a, (phi_status)
	inc	e
        bit	SAD_D1, a
	jr	nz, .handle_EOI         ; Possible EOI or SAD
	xor	a
	cp	e                       ; read 256 bytes?
	jr	nz, .loop

.handle_EOI:
        and	D0D1_m
	cp	SAD_D1_m                ; SAD received?
	jr	nz, .cmddone            ; no EOI, command input complete
	call	check_for_parity_error
	dec	hl
        ld	a, (hl)                 ; get the SAD byte we just read (not the original one)
	cp	HP300_CLEAR_SAD
	jp	z, HP_300_Clear_jv
        ; missing error handling for multiple SAD bytes received not done here?
        ; (Fall through ignores any non HP-300 Clear SAD)

.cmddone:
        pop	hl
	ld	(hl), e          ; save the transfer byte count
	pop	bc

	ld	hl, cmdtbl
	ld	d, 0
;x0228:
; Note: guaranteed to find a match due to the table lookup in read_Amigo_cmd.
.loopdispatch:
        ld	a, c		        ; does secondary address match table header?
	cp	(hl)
	jr	z, .found	        ; yes
	inc	hl		        ; no, get length of table
	ld	b, (hl)
	inc	hl

	xor	a		        ; HL := HL + b * 4 (find next table)
.mullp:	add	a, 4
	djnz	.mullp
	ld	e, a
	add	hl, de                  ; HL now points to the next table
	jr	.loopdispatch

.found:	inc	hl
	ld	a, (cmd_opcode)
	and	1fh                     ; mask parity?
	jr	z, .nounit              ; opcode == 00h only for Cold Load Read
	ld	b, a
	ld	a, (cmd_unit)
	and	0fh                     ; mask parity?
	ld	(cmd_unit), a
	ld	a, b                    ; A has opcode again

.nounit:
        ld	b, (hl)		        ; get length of table
	ld	e, 4		        ; DE = 4 (table entry length)
	inc	hl
.cmdlp:	cp	(hl)		        ; does command match?
	jr	z, .cmdfound
	add	hl, de		        ; no, advance to next entry
	djnz	.cmdlp
	jp	abort_illegal_opcode	; command not found

.cmdfound:
        inc	hl		        ; advance pointer to length byte
	ld	a, (cmd_count)
	cp	(hl)		        ; does command length match?
	jp	nz, abort_IO_program_error
	inc	hl		        ; advance pointer to dispatch vector
	pop	de
	jp	dispatch_cmd


; command dispatch tables
; a series of tables for each listen secondary address (08 through 0c)

; each table has a two byte header
;   first byte - secondary address
;   second byte - table length in 4 byte entries

; each table entry is four bytes
;   first byte - opcode
;   second byte - data byte count (including SAD byte)
;   third and fourth bytes - execution address

cmdtbl:
; secondary address 08
	db	08h, 09h

	db	03h, 02h
	dw	cmd_request_status

	db	02h, 06h
	dw	cmd_seek

	db	05h, 02h
	dw	cmd_unbuffered_read

	db	08h, 02h
	dw	cmd_unbuffered_write

	db	15h, 02h
	dw	cmd_end

	db	14h, 02h
	dw	cmd_request_logical_address

	db	07h, 04h
	dw	cmd_verify

	db	0bh, 02h
	dw	cmd_initialize

	db	00h, 02h
	dw	cmd_cold_load_read

; secondary address 09
	db	09h, 01h

	db	08h, 02h
	dw	cmd_buffered_write

; secondary address 0a
	db	0ah, 03h

	db	03h, 02h
	dw	cmd_request_status
	
	db	05h, 02h
	dw	cmd_buffered_read

	db	14h, 02h
	dw	cmd_request_logical_address

; secondary address 0b
	db	0bh, 02h

	db	05h, 02h
	dw	cmd_buffered_read_verify

	db	06h, 02h
	dw	cmd_id_triggered_read

; secondary address 0c
	db	0ch, 05h

	db	19h, 02h
	dw	cmd_door_lock

	db	1ah, 02h
	dw	cmd_door_unlock

	db	05h, 02h
	dw	cmd_unbuffered_read_verify

	db	018h, 005h
	dw	cmd_format

	db	14h, 02h
	dw	cmd_request_physical_address

check_for_holdoff:
        ld	a, (DSJ)
	cp	DSJ_holdoff
	ret	nz              ; not holdoff
	jp	cmd_terminate

;******************************************************************
; verify_known_format - verify the disc has a known format
;
; exits via abort_stat2_error if unknown
;x02c7:
verify_known_format:
        ld	a, (disc_format)
	cp	FORMAT_UNKNOWN_m
	jp	z, abort_stat2_error
	ret

verify_write_enabled:
        in	a, (drv_status)
	bit	WRPROT, a
	jp	nz, abort_stat2_error
	ret

; not quite check_unit_ready?? better term?
check_unit_ready:
        call	verify_unit_ready
	call	check_DSJ_S1
	ret	z                       ; OK, DSJ normal
	ld	a, (S1)
	cp	S1_ILL_OP
	ret	z
	cp	S1_IPE
	ret	z
	jp	cmd_terminate

;******************************************************************
; verify_unit_ready - validate the unit number and check for ready status
;
; on entry:
;       IY - RAM pointer
; 
; exits via cmd_terminate or abort_stat2_error if not valid/ready
; on exit:
;       IX: DriveData pointer
;

verify_unit_ready:
        ld	a, (iy + YB(cmd_unit))
	cp	NUNITS
	jp	m, .ok
	ld	a, S1_UA                ; unit unavailable
	call	set_abort_S1
	jp	cmd_terminate

.ok:	call	prep_drive_regs
	call	get_drive_SS
	jp	nc, abort_stat2_error
	jr	nz, .skip
	set	STAT2_F, (ix + DriveData.stat2_ds)
.skip:	bit	STAT2_F, (ix + DriveData.stat2_ds)
	jp	nz, abort_stat2_error
	ret


;******************************************************************
; do_HP_300_Clear - handle the HP-300 Device Clear command
;
; This is the expected HP-IB sequence for an HP-300 Device Clear:
; MLA SDC <control-byte> SDC UNL
;x0312:
do_HP_300_Clear:
        ld	bc, 0
	ld	d, 90                           ; wait ? ms for DEVICE CLEAR?
.loop:	in	a, (phi_int_cond)
	bit	DEV_CLR, a
	jr	nz, .gotdcl
	djnz	.loop
	dec	c
	jr	nz, .loop
	dec	d
	jr	nz, .loop

.clrerr:
        ld	a, DEV_CLR_m		        ; clear DEVICE CLEAR status not received
	out	(phi_int_cond), a
	jp	abort_IO_program_error

.gotdcl:
        bit	FIFO_BYTE, a
	jr	z, .clrerr                      ; control byte is missing

	in	a, (phi_fifo)
	ld	b, a
	in	a, (phi_status)	                ; get D0D1 for this byte
	and	D0D1_m
	cp	IN_EOI
	jr	nz, .clrerr	                ; not EOI

	xor	a                               ; A = DSJ_normal
	ld	(DSJ), a
	bit	ENABLE_PARITY_CHECK, b
	jr	z, .skip                        ; disable parity check (A == 0 here)

	ld	a, PRTY_FRZ_m                   ; enable parity check
.skip:	ld	b, a
	in	a, (phi_control)
	or	b
	out	(phi_control), a                ; set the desired parity freeze state

	ld	a, b
	ld	(parity_check_state), a         ; save the parity check state
	out	(phi_status), a
	ld	a, FIFO_ROOM | FIFO_BYTE | DEV_CLR
	out	(phi_int_mask), a
	bit	PRTY_FRZ, b                     ; parity check enabled?
	jp	nz, do_device_clear
	call	check_for_parity_error
	jp	do_device_clear

;******************************************************************
; saves AF or jumps to normal if no command_byte is waiting
;x0360:
do_check_for_command_byte:
        push	af
	call	do_PPD
	in	a, (phi_int_cond)
	bit	FIFO_BYTE, a
	jp	z, done_normal  ; no fifo bytes available, done_normal doesn't return to the caller
	pop	af
	ret

;******************************************************************
; cmd_end - wait for a change in drive status or a command is received
cmd_end:
	call	check_for_holdoff
	ld	a, IU_WAIT
	set_in_use_leds
	ld	a, (current_unit)
	ld	(tmp_unit), a
	call	normal_and_deselect_drives

.scanloop:
        ld	(iy + YB(current_unit)), 3      ; Scan the drives for a change
.nextdrive:
        ld	a, (current_unit)
	call	prep_drive_regs
	ld	e, (ix + DriveData.stat2_ds)
	call	get_drive_SS
	jr	nz, .skip                       ; A not valid (no SS value)
	jr	c, .gofs                        ; drive ready
	ld	b, a                            ; B = current SS value
	ld	a, e                            ; A = previous SS value
	and	SS_m
	cp	SS_NO_DRV
	jr	z, .skip
	cp	b
	jr	nz, .goda                       ; SS value has changed
.skip:	bit	STAT2_F, e
	jr	nz, .gofs
	call	poll_input_or_DCL
	jr	nz, .exitend
	call	deselect_drives
	dec	(iy + YB(current_unit))
	jp	p, .nextdrive

	ld	b, 0                            ; delay a bit looking for new input
.loop:	call	poll_input_or_DCL
	jr	nz, .exitend
	djnz	.loop

	ld	d, 18                           ; approx 18/256s elapsed?
	call	head_unload_timer_check
	jr	.scanloop

.gofs:	set	STAT2_F, e
.goda:	set	STAT2_A, e
	ld	(ix + DriveData.stat2_ds), e
	ld	a, S1_DA
	call	set_abort_S1

.exitend:
        call	deselect_drives
	ld	a, (tmp_unit)
	ld	(current_unit), a
	ret


	fillto	03e0h, 076h


        ; return the DSJ byte
cmd_DSJ:
        in	a, (phi_int_cond)
	in	a, (phi_status)
	and	PRTY_ERR_m
	jr	nz, .prtyerr

.continue:
        in	a, (phi_control)
	or	INIT_FIFO_m
	out	(phi_control), a

	ld	a, OUT_END_m | DATA_FRZ_m       ; EOI, initialize outbound FIFO
	out	(phi_status), a
	ld	a, (DSJ)                        ; send the current DSJ byte
	out	(phi_fifo), a
	in	a, (phi_int_cond)
	bit	FIFO_BYTE, a
	jp	nz, listen_terminate            ; unexpected byte in the inbound FIFO
	ld	a, (DSJ)
	sub	DSJ_holdoff                     ; detect transition from holdoff or parity error?
	ret	c                               ; return if DSJ was normal or abort

        ; Map the last command's S1 code into the previous DSJ value
	ld	a, (S1)
	or	a
	jr	z, .saveit                      ; DSJ_normal
	cp	S1_DA                           ; drive attention?
	jr	nz, .aborted
	ld	b, (iy + YB(current_unit))
	call	get_drive_data_ptr_in_ix
	call	get_drive_stat2_bytes
	bit	STAT2_E, d
	jr	nz, .aborted
	bit	STAT2_C, d
	ld	a, DSJ_normal
	jr	z, .saveit
.aborted:
        ld	a, DSJ_aborted
.saveit:
        ld	(DSJ), a
	ret

.prtyerr:
        ld	b, PRTY_ERR             ; reset the parity error
	out	(phi_status), a
	xor	a
	out	(phi_int_cond), a
	ld	(iy + YB(DSJ)), DSJ_parity_error
	jr	.continue


cmd_request_status:
	call	check_for_holdoff
	ld	a, IU_STAT
	set_in_use_leds
	ld	a, (cmd_unit)
	call	determine_unit_status_jv

;******************************************************************
; do_send_status_or_address - send the request status or address response bytes
;       waits for the talk 08h SAD then send the data bytes
;
; on entry:
;       C = first byte, B = second byte
;       E = third byte, D = fourth byte
;
; For status:
;       C: stat1, B: stat1_unit
;       E: stat2, D: stat2_ds
;
; For addresses:
;       C: cylinder high, B: cylinder low
;       E: head, D: sector or zeros

do_send_status_or_address:
        push	bc                              ; save the status bytes
	push	de
	ld	b, TLK_M | 08h
	call	wait_for_HPIB_SAD               ; (doesn't return if an error)
	pop	de                              ; restore status buffer bytes
	pop	bc
	ld	hl, cmd_count
	ld	(stat1), bc                     ; C: stat1, B: stat1_unit
	ld	(stat2), de                     ; E: stat2, D: stat2_ds
	ld	(iy + YB(stat2_ds+1)), 1        ; dummy byte for the EOI?
	ld	b, FLUSH_FIFO_m | SEND_EOI_m
	ld	a, 4                            ; transfer 4 bytes
	call	do_talk_transfer
	call	normal_and_deselect_drives
	ret

do_determine_unit_status:
        ld	b, a                    ; save unit in B
	cp	NUNITS
	jr	c, .unitvalid
	ld	c, S1_UA
	ld	de, 0                   ; no stat 2 bits for unit unavailable
	ret

        ; build status data values
        ; on return:
	; C = stat1:            00DSSSSS
        ; B = stat1_unit:       000000UU
	; E = stat2:            *??TTTTR
        ; D = stat2_ds: AW?EFCSS
.unitvalid:
        call	prep_drive_regs
	call	get_drive_stat2_bytes           ; D = stat2_ds, E = stat2
	call	get_drive_SS
	jr	z, .save_ss                     ; Z -> A is valid, A = SS bits
	ld	a, SS_READY
	bit	STAT2_F, d
	jr	nz, .save_ss
	ld	a, d                            ; A = previous stat2_ds
	and	SS_m                            ; isolate the SS field
	cp	SS_NO_DRV                       ; is a drive connected?
	scf
	jr	nz, .jrsr                       ; yes

.save_ss:
        push	af                              ; save flags from get_drive_SS
	cp	SS_NO_DISC                      ; no disc in drive?
	jr	nz, .skipnd
	ld	a, d
	and	SS_m
	cp	SS_NO_DRV                       ; is a drive connected?
	jr	z, .skipnd
        ld	a, SS_NO_DISC
.skipnd:
        ld	b, a
	pop	af                              ; restore flags from get_drive_SS
	ld	a, b                            ; A = SS (drive status bits)
	res	SS_NOT_READY, d                 ; SS = 00 (drive ready)
	res	0, d                            ; (clear SS_UNDEF just in case?)
	ld	e, 0                            ; clear stat2 fields
	res	STAT2_UNDOC, d                  ; ?
	res	STAT2_W, d
	jr	c, .first                       ; (C saved from get_drive_SS) first status
	or	d                               ; include the SS bits
	ld	d, a
.jrsr:	jr	.setregs

.first:	set	STAT2_F, d                      ; (first status)
	in	a, (drv_status)
	bit	WRPROT, a
	jr	z, .skipw                       ; no
	set	STAT2_W, d                      ; (write protect)
.skipw:	call	load_head_ix                    ; try to determine the disc type?
	res	HEAD1, (ix + DriveData.drv)     ; Select head 0
	deassert_in_use
	call	get_double_sided_bit
	call	media_type_check
	jr	z, .setregs
	ld	a, S1_S2E                       ; Stat 2 error
	call	set_abort_S1
	set	DT_BLANK, e
.setregs:
        ld	c, (iy + YB(S1))                ; S1
	ld	b, (iy + YB(S1_unit))           ; S1_unit
	ld	a, d
	and	STAT2_E_m | STAT2_C_m | SS_m
	jr	z, .skipe
	set	STAT2_ERROR, e
.skipe:	ld	a, d
	and	STAT2_W_m | STAT2_UNDOC_m | SS_m ; clear error bits
	ld	(ix + DriveData.stat2_ds), a
	ld	a, e
	and	~STAT2_ERROR
	ld	(ix + DriveData.stat2), a
	jp	deselect_drives                 ; returns from deselect_drives


	fillto	04f0h, 076h


;******************************************************************
; cmd_seek - perform the seek command
;
; on entry:
;       IX - DriveData pointer
;       IY - RAM pointer

cmd_seek:
	call	check_for_holdoff
	call	verify_unit_ready
	call	verify_known_format
	ld	d, (iy + YB(seek_head))
	ld	b, (iy + YB(seek_sector))
	ld	a, (iy + YB(seek_cylinder_msb))
	or	a
	jp	nz, abort_seek_check
	ld	a, (iy + YB(seek_cylinder_lsb))
	ld	c, a
	cp	0
	jp	m, abort_seek_check
	cp	PHYSICAL_TRACKS
	jp	nc, abort_seek_check
	call	validate_sector_number
	jp	nz, abort_seek_check
	xor	a
	cp	d                                       ; D: head
	jr	z, .hdok
	inc	a
	cp	d
	jp	nz, abort_seek_check
	bit	DT_DS, (ix + DriveData.stat2)
	jp	z, abort_seek_check

.hdok:  ld	a, d
	rrca                                            ; move head number into bit 7
	or	c                                       ; combine cylinder and head
	ld	c, a 
	push	bc
	load_head
	pop	bc
	ld	(ix + DriveData.target_cyl_hd), c
	ld	(ix + DriveData.sector), b
	ld	a, (ix + DriveData.current_cyl_hd)
	cp	c
	jr	z, .success
	wait_for_sector                                 ; see where we're at now
	cp	(ix + DriveData.current_cyl_hd)
	jr	z, .call_seek                           ; the head is where we expect
	ld	b, (ix + DriveData.target_cyl_hd)       ; not at the expected cylinder, rezero it
	ld	c, (ix + DriveData.sector)
	push	bc
	reset_drive
	pop	bc
	ld	(ix + DriveData.target_cyl_hd), b
	ld	(ix + DriveData.sector), c
	jp	nz, abort_drive_fault_seek

;x0555:
.call_seek
        seek                                            ; calls do_seek below via the rst jv
.success:
        set	STAT2_A, (ix + DriveData.stat2_ds)
	ld	a, S1_DA                                ; drive attention
	call	set_abort_S1                            ; a "successful abort"
	ld	(iy + YB(DSJ)), DSJ_normal              ; indicate success
	call	deselect_drives
	ret

do_seek:
	call	do_seek_to_cylinder
	ret	z
	jp	c, abort_drive_fault_seek
	ld	b, (ix + DriveData.sector)
	ld	c, (ix + DriveData.target_cyl_hd)
	push	bc
	cp	1                                       ; did do_seek_to_cylinder return A == 1?
	call	z, reset_drive_jv
	pop	bc
	ld	(ix + DriveData.sector), b
	ld	(ix + DriveData.target_cyl_hd), c
	jp	abort_seek_check


;******************************************************************
; do_seek_to_cylinder - seek to the desired cylinder and head
;
; on entry:
;       IX - DriveData pointer
;       IY - RAM pointer
;       DriveData.target_cyl_hd - the desired cylinder and head
;
; on exit:
;       C - error
;       Z - success?

do_seek_to_cylinder:
        ld	a, IU_STEP
	set_in_use_leds
	ld	(iy + YB(tmp_unit)), 10                 ; max 10 seek retry attempts
	ld	a, (ix + DriveData.target_cyl_hd)
	or	a
	jr	nz, perform_seek
	ld	b, (ix + DriveData.sector)              ; save sector across call to reset_drive
	push	bc
	reset_drive                                     ; use reset_drive to seek to cylinder 0
	pop	bc
	ld	(ix + DriveData.sector), b
	jr	z, .at_track0
	scf
	ret                                             ; C, NZ - indicate error

.at_track0:
        ld	c, 0
	ld	b, c
	ld	d, (ix + DriveData.drv)
	jr	stepit

perform_seek:
        ld	a, (ix + DriveData.current_cyl_hd)
	xor	(ix + DriveData.drv)
	bit	HEAD1, a            
	push	af                                      ; remember if a head change is needed
	ld	a, (ix + DriveData.current_cyl_hd)
	and	BYTENOT(HEAD1_m)
	ld	b, a
	ld	a, (ix + DriveData.target_cyl_hd)
	and	BYTENOT(HEAD1_m)
	sub	b
	ld	b, a                                    ; B: number of tracks and direction to step
	pop	af
	jr	z, .sethd                               ; no head change needed
	ld	a, (ix + DriveData.current_cyl_hd)      ; adjust B appropriately for the
	xor	(ix + DriveData.target_cyl_hd)          ; current cylinder and the target combos
	bit	HEAD1, a
	jr	z, .sethd
	ld	a, (ix + DriveData.target_cyl_hd)
	inc	b
	bit	HEAD1, a
	jr	nz, .sethd
	dec	b
	dec	b
.sethd:	ld	d, (ix + DriveData.drv)                 ; swap heads if needed
	ld	a, (ix + DriveData.current_cyl_hd)
	xor	(ix + DriveData.target_cyl_hd)
	bit	HEAD1, a
	jr	z, .notoggle
	bit	HEAD1, d                                ; toggle HEAD1 in D
	res	HEAD1, d
	jr	nz, .notoggle
	set	HEAD1, d
.notoggle:
        xor	a
	cp	b
	ld	c, b
	jr	nz, stepit
	bit	HEAD1, d
	jr	nz, stepit
	bit	DT_DS, (ix + DriveData.stat2)
	jr	z, stepit                               ; single sided
	ld	c, -1                                   ; stepping out towards 0
stepit:	call	do_steps
	jr	nz, .x0661
	res	HEAD1, (ix + DriveData.drv)
	bit	HEAD1, d
	jr	z, .skip
	set	HEAD1, (ix + DriveData.drv)
.skip:	deassert_in_use
	ld	b, 12
.retry_wfs:
        push	bc
	wait_for_sector
	pop	bc
	cp	0ffh                                    ; A: 0ffh -> failed
	jp	nz, .verify_seek
	dec	b
	jr	z, .x0663
	bit	DT_DS, (ix + DriveData.stat2)
	jr	z, .x063b                               ; single sided
	bit	HEAD1, (ix + DriveData.drv)
	set	HEAD1, (ix + DriveData.drv)
	jr	z, .x062c
	res	HEAD1, (ix + DriveData.drv)             ; select head 0
.x062c:	deassert_in_use
	xor	a                                       ; maybe select head 0
	bit	7, c                                    ; seeking towards 0 (sign bit set)?
	jr	z, .x0634
	ld	a, HEAD1_m                              ; select head 1
.x0634:	xor	(ix + DriveData.drv)                    ; reset drv_shdw
	bit	HEAD1, a
	jr	nz, .retry_wfs
.x063b:	push	bc
	ld	b, 1
	bit	7, c                                    ; seeking towards 0 (sign bit set)?
	jr	z, .x0644
	ld	b, -1
.x0644:	call	do_steps
	pop	bc
	jr	nz, .x0661
	jr	.retry_wfs

.verify_seek:
        ld	(ix + DriveData.current_cyl_hd), a      ; current cylinder in A from wait_for_sector
	cp	(ix + DriveData.target_cyl_hd)
	call	z, show_previous_in_use_leds            ; we are at the target cylinder
	ret	z
	dec	(iy + YB(tmp_unit))                     ; oops, didn't get there
	jp	nz, perform_seek                        ; retry
	or	1                                       ; set NZ
	ld	a, 0
	ret

.x0661:	ld	a, 1
.x0663:	or	a                                       ; set NZ
	ret


;x0665:
do_steps:
        ld	a, b
	or	a
	ret	z                                       ; no cylinder change
	jp	p, .seek_in
	add	a, (ix + DriveData.physical_cylinder)   ; seeking out
	cp	0
	jp	m, .skip                                ; would go beyond cylinder 0
	in	a, (drv_status)
	bit	TRACK0, a                               ; at cylinder 0?
.skip:	ret	nz
	jr	.dosteps

.seek_in:
        add	a, (ix + DriveData.physical_cylinder)
	cp	PHYSICAL_TRACKS
	jp	m, .dosteps
	inc	a                                       ; at the highest cylinder
	ret

.dosteps:
        push	bc
	xor	a
	ld	c, b                                    ; C: skip direction
	cp	b
	jp	m, .steploop
	ld	a, b                                    ; form abs(B) for loop count
	neg
	ld	b, a
.steploop:
        call	step_drive_jv
	bit	7, c                                    ; seeking in or out?
	jr	z, .skipdec
	dec	(ix + DriveData.physical_cylinder)
	jr	.continue

.skipdec:
        inc	(ix + DriveData.physical_cylinder)
.continue:
        djnz	.steploop
	ld	b, 20
	call	delay                           ; delay 20 ms
	pop	bc
	xor	a
	ret

; do_step_drive - step drive one track
;
; on entry:
;       C: negative => step out
;
; NOTE: The drive document says 3 ms (min) required between step pulses
;       and 18 ms between step pulses if a direction change has taken place
;       with no intervening read/write operation performed. The step pulse
;       should be 1 us min width.

do_step_drive:
        push	bc
	ld	a, (ix + DriveData.drv)
	and	~HEADLOAD               ; unload_head now?
	bit	STEP_OUT, c
	jr	nz, .skip
	set	MOVEIN, a               ; It appears DriveData.drv keeps MOVEIN deasserted
.skip:	out	(drv), a                ; set MOVEIN as required
	set	STEP, a                 ; Pulse STEP
	out	(drv), a
	res	STEP, a
	out	(drv), a
	deassert_in_use
	ld	b, 3                    ; Delay 3 ms
	call	delay
	pop	bc
	ret

;******************************************************************
; do_reset_drive - "recalibrate drive"
;       reset current addresses and seek to track 0
;
; on exit:
;       Z: at track 0, A = 0
;      NZ: error

do_reset_drive:
	xor	a
	ld	(ix + DriveData.physical_cylinder), a
	ld	(ix + DriveData.target_cyl_hd), a
	ld	(ix + DriveData.current_cyl_hd), a
	ld	a, (disc_format)
	and	FORMAT_IBM_m
	ld	(ix + DriveData.sector), a
	res	HEAD1, (ix + DriveData.drv)     ; select head 0
	deassert_in_use

	ld	b, 80                           ; step to track 0
.step_loop:
        in	a, (drv_status)
	bit	TRACK0, a
	jr	nz, .at_track0
	ld	c, 0ffh                         ; step towards track 0
	call	step_drive_jv
	djnz	.step_loop
	inc	b                               ; NZ (not at track 0 after 80 step attempts)
	ret

.at_track0:
        ld	a, 80
	cp	b
	ld	b, 20
	call	nz, delay                       ; delay 20 ms if we weren't at track 0 already
	xor	a
	ret                                     ; Z

	fillto	0700h, 076h

cmd_buffered_read_verify:
	call	sio_common_prep
	set	MGNENA, (ix + DriveData.drv)     ; turn on reduced margins
	jr	buffered_read_common

cmd_buffered_read:
	call	sio_common_prep
buffered_read_common:
        load_head
	call	head_unload_timer_1s_check
	ld	hl, 0f073h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | CRCERR_m | SIO_E_NOT_READY_m | SIO_E_1_m | SIO_E_SCE_m)
	ld	a, 1
	call	read_the_sector
;x0718
do_send_data:
        push	af
	ld	b, TLK_m | 00h
	call	wait_for_HPIB_SAD               ; wait until we're told to talk
	ld	b, 3
	pop	af
	call	do_read_send_data               ; do_read_send_data doesn't alter flags!
	jp	z, read_success                        ; success from read_the_sector
	jp	read_failed                           ; failure

sio_common_prep:
        call	check_for_holdoff
	call	check_unit_ready
	call	verify_known_format
verify_cylinder:
        call	check_cylinder_in_range
	jp	nc, abort_seek_check
	ret

cmd_unbuffered_read_verify:
	call	sio_common_prep
	set	MGNENA, (ix + DriveData.drv)
	jr	unbuffered_read_common

cmd_unbuffered_read:
	call	sio_common_prep

;x0746
unbuffered_read_common:
        ld	hl, 0f073h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | CRCERR_m | SIO_E_NOT_READY_m | SIO_E_1_m | SIO_E_SCE_m)
	push	hl
	load_head
	ld	a, 1
	call	read_the_sector
	push	af
	ld	b, TLK_m | 00h
	call	wait_for_HPIB_SAD
	ld	b, 2
	pop	af
	jr	z, .success
	ld	b, 3                            ; B bit 1 = 1: flush out fifo
.success:
        call	do_read_send_data
	jp	nz, read_failed
.again:	call	poll_input_or_DCL
	pop	hl
	jp	nz, read_success
	in	a, (phi_status)
	bit	TLK_IDF, a                      ; Still addressed to talk?
	jp	z, read_success                 ; no, done
	call	head_unload_timer_1s_check
	deassert_in_use
	xor	a
	call	read_the_sector
	push	hl
	ld	b, 0
	jr	z, .skip
	ld	b, 1
.skip:	push	af
	call	do_read_send_data
	pop	af
	jp	nz, read_failed
	jr	.again

cmd_verify:
	ld	a, IU_VFY
	set_in_use_leds
	call	check_for_holdoff
	call	verify_unit_ready
	call	verify_known_format
	call	verify_cylinder_jv
	ld	a, (verify_count_msb)
	cp	0
	jp	m, abort_IO_program_error
	ld	b, (iy + YB(verify_count_msb))
	ld	c, (iy + YB(verify_count_lsb))
	push	bc
	ld	a, b
	or	a
	jr	z, .x07b2
	cp	1
	jr	c, .x07c8
	jr	.x07b5

.x07b2:	add	a, c
	jr	nz, .x07c8
.x07b5:	ld	b, 4
	push	ix
.x07b9:	dec	b
	ld	a, b
	jp	m, .x07c6
	cp	(iy + YB(current_unit))
	call	nz, maybe_unload_head
	jr	.x07b9

.x07c6:	pop	ix
.x07c8:	load_head
	pop	bc
	set	MGNENA, (ix + DriveData.drv)
	ld	d, 1
.again:	ld	hl, 0f073h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | CRCERR_m | SIO_E_NOT_READY_m | SIO_E_1_m | SIO_E_SCE_m)
	deassert_in_use
	push	bc
	ld	a, d
	call	read_the_sector
	jp	nz, read_failed
	pop	bc
	ld	d, 0
	cpi
	jp	pe, .again

read_success:
        call	normal_and_deselect_drives
read_cleanup:
        res	MGNENA, (ix + DriveData.drv)
	deassert_in_use
	ret

read_failed:
        call	read_cleanup
	jp	listen_terminate


; cmd_cold_load_read - Read from unit 0 cylinder 0 at specified head and sector
;                      until the controller terminates the I/O.
cmd_cold_load_read:
	xor	a                               ; DSJ_normal, unit 0
	ld	(DSJ), a
	call	determine_unit_status_jv
	xor	a
	call	prep_drive_regs
	call	verify_known_format
	ld	a, (iy + YB(clr_hd_sect))
	and	3fh                             ; isolate the sector field
	ld	b, a
	call	validate_sector_number
	jp	nz, abort_seek_check
	ld	a, (iy + YB(clr_hd_sect))
	rlca                                    ; isolate head field
	rlca
	and	3
	jr	z, .ok
	cp	2
	jp	nc, abort_seek_check            ; only 2 heads on 9895A
	bit	3, (ix + DriveData.stat2)
	jp	z, abort_seek_check
	rrca
.ok:	ld	(ix + DriveData.target_cyl_hd), a
	ld	(ix + DriveData.sector), b
	jp	unbuffered_read_common


cmd_id_triggered_read:
	call	sio_common_prep
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jp	nz, abort_stat2_error           ; not supported for IBM format discs?
	load_head
	call	seek_to_target_cyl_hd
	jr	z, .doread
	jp	c, abort_drive_fault
	jp	nc, abort_seek_check
.doread:
        ld	hl, 0f073h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | CRCERR_m | SIO_E_NOT_READY_m | SIO_E_1_m | SIO_E_SCE_m)
	call	wait_for_sector_id_field
	call	do_sector_io_check
	jr	nc, .doread
	jp	nz, cmd_terminate
	ld	b, 2
	call	delay                           ; delay 2 ms
	ld	a, 6                            ; # retry attampts?
	call	do_read_sector_common
	call	do_sector_io_check
	jr	nc, .doread
	call	z, next_logical_sector_save_flags
	jp	do_send_data

; on exit:
;       Z: success
;       NZ: failure
read_the_sector:
        push	af
.again:	call	seek_to_target_cyl_hd
	jr	nz, rts_err
	call	do_read_sector
	call	do_sector_io_check
	jr	nc, .again
	inc	sp                      ; pop one return address, saving regs
	inc	sp
	ret	nz

next_logical_sector_save_flags:
        push	af
	call	next_logical_sector
	pop	af
	ret

rts_err:
        jr	c, .sete
	cp	1
	jr	nz, .setc
	pop	af
	push	af
	or	a
	jr	nz, .setc
	call	normal_and_deselect_drives
	jr	.doexit

.setc:	set	STAT2_C, (ix + DriveData.stat2_ds)
	jr	.seta

.sete:	set	STAT2_E, (ix + DriveData.stat2_ds)
.seta:	set	STAT2_A, (ix + DriveData.stat2_ds)
	ld	a, S1_DA                        ; drive attention
	call	set_abort_S1
.doexit:
        pop	af
	ld	a, 1
	or	a                               ; set NZ
	ret

check_cylinder_in_range:
        ld	a, (ix + DriveData.target_cyl_hd)
	and	BYTENOT(HEAD1)
	cp	PHYSICAL_TRACKS
	ret

seek_to_target_cyl_hd:
        ld	a, (ix + DriveData.current_cyl_hd)
	cp	(ix + DriveData.target_cyl_hd)
	ret	z
	push	hl
	ld	b, (ix + DriveData.drv)
	push	bc                              ; Save DriveData.drv value
	call	seek_to_cylinder_jv             ; This may affect MGNENA?
	pop	bc
	push	af                              ; save A value?
	ld	a, b
	and	MGNENA_m                        ; Isolate MGNENA from DriveData.drv copy
	or	(ix + DriveData.drv)            ; or in current drv_shdw (changed by do_seek_to_cylinder?)
	ld	(ix + DriveData.drv), a         ; update it
	deassert_in_use
	pop	af
	pop	hl
	ret

; do_sector_io_check - test for errors during sector I/O and set statuses
;
; on entry:
;       E: result of sio_done_get_status
;
; on exit:
;       A: unchanged - not sure why

do_sector_io_check:
        push	af                      ; save A
	ld	b, S1_S2E               ; check stat2 for cause
	bit	SIO_E_NOT_READY, e
	jr	nz, dsic_abort
	ld	b, S1_SCE               ; sector compare error
	bit	SIO_E_SCE, e
	jr	nz, dsio_retry_if_sector_was_found
	ld	b, S1_CCE               ; cylinder compare error
	bit	SIO_D_WRONG_TRACK, d
	jr	nz, dsio_retry_seek
	ld	b, S1_SCE               ; sector compare error
	bit	SIO_D_WRONG_HEAD, d
	jr	nz, dsio_retry_seek
	ld	b, S1_DBIT_m | S1_DTE
	bit	SIO_E_DEFECTIVE_SECTOR, e
	jr	nz, dsic_abort
	ld	b, S1_SCE               ; sector compare error
	bit	SIO_E_1, e
	jr	nz, dsic_abort
	ld	a, e
	and	l                       ; oh noes, what is L here?
	ld	b, S1_RHE
	bit	OVERUN, a               ; couldn't keep up, retryable hardware error
	jr	nz, dsio_retry_seek
	ld	b, S1_UDE               ; uncorrectable data error
	bit	CRCERR, a
	jr	z, dsic_success
	bit	MGNENA, (ix + DriveData.drv)
	jr	z, dsic_abort           ; not using reduced read margins
	bit	SIO_D_WFS_FAILED, d
	jr	z, dsic_abort
	ld	b, S1_SCE               ; sector compare error
	jr	dsic_abort

dsic_success:
        xor	a
dsic_exit:
        res	SECTOR_IO_RETRY, (iy + YB(state_flags))
	scf
dsic_cleanup:
        ex	(sp), hl                ; AF pushed at entry
	ld	a, h                    ; recover A
	pop	hl                      ; get back HL
	ret

dsic_abort:
        ld	a, b
	call	set_abort_S1
	or	1                       ; exit with A = 1/NZ
	jr	dsic_exit

dsio_retry_seek:
        bit	SECTOR_IO_RETRY, (iy + YB(state_flags))
	jr	nz, dsic_abort                  ; only retry once
	ld	a, (ix + DriveData.drv)         ; drv_shdw?
	push	af                              ; Remember DriveData.drv state for later
	push	hl
	ld	a, (ix + DriveData.target_cyl_hd)
	push	af
	ld	(ix + DriveData.target_cyl_hd), 0
	seek                                    ; rezero drive
	pop	af                              ; Get back saved (ix + DriveData.target_cyl_hd)
	or	a                               ; Any bits set in the cylinder number?
	jr	z, .skipseek                    ; no, already at cylinder 0
	ld	(ix + DriveData.target_cyl_hd), a
	seek
.skipseek:
        pop	hl
	pop	af                              ; Get back saved DriveData.drv
	and	MGNENA_m                        ; Isolate MGNENA
	or	(ix + DriveData.drv)            ; Combine previous MGNENA with drv reg copy
	ld	(ix + DriveData.drv), a         ; udpate it
	deassert_in_use
	set	SECTOR_IO_RETRY, (iy + YB(state_flags))
	or	1                               ; exit with A = 1/NZ
	jr	dsic_cleanup

dsio_retry_if_sector_was_found:
        bit	SIO_D_SECTOR_FOUND, d
	jr	z, dsio_retry_seek
	jr	dsic_abort

;
; on entry:
;       A = 0: send data buffer
;       A = 1: send end of transfer byte with EOI?
;       B bit 1 = 1: flush the out fifo first
do_read_send_data:
        push	af
	ld	hl, data_buffer_count
	cp	1
	jr	z, .sendEOIbyte
	call	do_talk_transfer
	pop	af
	ret

.sendEOIbyte:
        bit	1, b
	call	nz, flush_out_fifo
	ld	a, 1             ; placeholder byte?
	ld	b, OUT_END_m     ; D0D1 = 10 (EOI)
	call	send_out_byte
	pop	af
	ret

; next_logical_sector - move to the next logical sector
;
; on entry:
;       DriveData contains the current state
;
; on exit:
;       DriveData contains the updated state

next_logical_sector:
        ld	b, (ix + DriveData.sector)
	inc	b
	call	validate_sector_number
	jr	z, .sector_valid
	bit	DT_DS, (ix + DriveData.stat2)                   ; done with this track
	jr	z, .skipds
	bit	HEAD1, (ix + DriveData.target_cyl_hd)           ; toggle the head
	set	HEAD1, (ix + DriveData.target_cyl_hd)
	jr	z, .skiph1                                      ; same cylinder, now HEAD1
	res	HEAD1, (ix + DriveData.target_cyl_hd)
.skipds:
        inc	(ix + DriveData.target_cyl_hd)                  ; was HEAD1, next cylinder
.skiph1:
        ld	b, (iy + YB(disc_format))                       ; get the starting sector number
.sector_valid:
        ld	(ix + DriveData.sector), b
	ret


; do_wait_for_sector - wait for a sector to be found on the track
;
; on exit:
;       NZ: failed, A: -1, B: ?
;       Z:  success, A: current cylinder, B: current sector
do_wait_for_sector:
	in	a, (drv_status)
	bit	READY, a
	jp	z, abort_drive_fault_seek
	res	MGNENA, (ix + DriveData.drv)    ; give us the best chance to see where we are
	deassert_in_use
	ld	hl, (SIO_D_BIT4 << 8) | (OVERUN_m | CRCERR_m)
	ld	b, 2                            ; try two times
.loop:	push	bc
	call	wait_for_sector_id_field
	jr	z, .ret
	pop	bc
	djnz	.loop                           ; something failed, try again
	or	0ffh                            ; A = 0ffh, NZ: failed
	ret

.ret:	pop	hl                              ; success, drop BC off the stack
	ld	a, b                            ; A: current_cyl_hd?
	ld	b, c                            ; b: ?
	ret

; wait_for_sector_id_field
;
; NOTE: HP format sector number has the head number in bit 7
;
; on entry:
;       H: error mask for SIO_D_* flags
;       L: error mask for SIO_E_* flags
;       IX: DriveData pointer
;
; on exit:
;       B: current cylinder as read from the disc
;       C: sector number as read from the disc
;       D: SIO_D_* flags
;       E: SIO_E_* flags
;       NZ: failed
;       Z: success

;x09b5:
wait_for_sector_id_field:
        ld	a, r_OVERUN_m | r_TIMEOUT_m
	out	(reset), a                      ; reset error flags
	ld	a, (ix + DriveData.target_cyl_hd)
	and	BYTENOT(HEAD1_m)                ; just the cylinder
	ld	(target_cyl_hd), a
	ld	de, 0                           ; status flags word
again:	push	hl                              ; save error mask flags
	ld	a, d
	and	SIO_D_SECTOR_FOUND_m            ; indicate second attempt? AM found?
	ld	d, a
	ld	e, 0                            ; reset the E part of the flags
	push	de
wfs_id_am:
        pop	de                              ; DE: some type of operation flags
	xor	a
	out	(cntl), a                       ; preparation to find an AM
	ld	a, r_OVERUN_m
	out	(reset), a
	push	de
	ld	a, CRCON_m                      ; enable CRC checking
	ld	b, READON_m | CRCON_m           ; send data to CRC chip
	ld	c, switches + NOSW              ; (will read just the AMDT and TIMEOUT bits)
	ld	d, HP_ID_AM_DPAT
	ld	e, HP_AM_CLK
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jp	nz, wfs_ibm
	out	(cntl), a

        ; test for HP format?
wfs_hp:
        in      (c)                             ; undocumented Z80: in flags, (c)
	jr	z, wfs_hp                       ; wait until either AMDT or TIMEOUT are set
	jp	p, wfs_timedout
	in	a, (data)                       ; AMDT signal is asserted here
	ld	h, a                            ; save the AM byte
	in	a, (clock)
	cp	e                               ; does the clock pattern match?
	jr	nz, wfs_id_am                   ; nope
	ld	a, b
                                                ; NOTE: HP AMs not included in CRC calcs
	out	(cntl), a                       ; READON_m | CRCON_m from above
	in	a, (data)
	ld	b, a                            ; B: track number
	ld	a, h                            ; get back the AM byte
	sub	d                               ; D: HP_ID_AM_DPAT
	rl	a                               ; 50-70=e0 ->  C, c1  (HP_DATA_AM_DPAT)
                                                ; 70-70=00 -> NC, 00  (HP_ID_AM_DPAT)
                                                ; d0-70=60 -> NC, c0  (HP_ECC_DATA_AM_DPAT)
                                                ; f0-70=80 ->  C, 00  (HP_DEFECTIVE_TRACK_AM_DPAT)
	jr	nz, wfs_id_am                   ; HP_DATA_AM_DPAT or HP_ECC_DATA_AM_DPAT?
	in	a, (data)                       ; A: sector number
	pop	de                              ; DE is zeros first time around?
	push	af                              ; save flags and sector number
	in	a, (data)                       ; A: first crc byte
	set	SIO_D_SECTOR_FOUND, d
	call	verify_crc                      ; E has OVERUN_m | CRCERR_m | ~READY_m
	pop	af                              ; restore flags
	jr	nc, .skipdt                     ; NC => HP_ID_AM_DPAT
	set	SIO_E_DEFECTIVE_SECTOR, e       ; found a HP_DEFECTIVE_TRACK_AM_DPAT
;x0a13:
.skipdt:
        bit	HEAD1, a                        ; A: sector number, is this HEAD1?
	res	HEAD1, a                        ; clear the HEAD1 bit from A
	ld	c, a                            ; C: sector number field
	jr	z, .skiphead
	set	HEAD1, b                        ; set HEAD1 bit, B: cyl_hd
.skiphead:
        cp	(ix + DriveData.sector)         ; at the target sector?
	jr	z, .skipws                      ; yes
	set	SIO_D_WRONG_SECTOR, d
.skipws:
        ld	a, b
	and	BYTENOT(HEAD1_m)
	cp	(iy + YB(target_cyl_hd))        ; reached the target_cyl_hd?
	jr	z, .skipwt                      ; yes
	set	SIO_D_WRONG_TRACK, d
.skipwt:
        ld	a, b
	xor	(ix + DriveData.target_cyl_hd)
	bit	HEAD1, a
	jr	z, wfs_coda
	set	SIO_D_WRONG_HEAD, d
;x0a37
wfs_coda:
        ld	a, e                            ; E: verify_crc drv_status bits and accumulated bits
	pop	hl                              ; get back ERROR masks
	and	l                               ; mask from do_wait_for_sector L=60h
	jr	nz, .wfs_error
	ld	a, d
	and	h                               ; mask from do_wait_for_sector H=10h
	jr	nz, .wfs_error
	ret

.wfs_error:
        ld	a, e
	and	OVERUN_m | CRCERR_m
	jp	nz, again                       ; try again in case is recoverable transient
	ld	a, d
	and	SIO_D_WRONG_TRACK_m | SIO_D_WRONG_HEAD_m
	ret	nz
	ld	a, d
	and	SIO_D_WRONG_SECTOR_m | SIO_D_BIT4
	jp	nz, again
wfs_done:
        or	1                               ; set NZ
	set	SIO_D_WFS_FAILED, d
	ret

wfs_timedout:
        xor	a
	out	(cntl), a
	pop	de
	pop	hl
	set	SIO_E_SCE, e
	jr	wfs_done

; wfs_ibm - wait for an IBM format sector ID AM
;x0a5f:
wfs_ibm:     
        out	(cntl), a                       ; CRCON_m from above
.waitamibm:
        in      (c)                             ; undocumented Z80: in flags, (c)
	jr	z, .waitamibm                   ; wait until either AMDT or TIMEOUT are set
	jp	p, wfs_timedout
	ld	a, b
                                                ; NOTE: IBM AM is included in CRC calcs
	out	(cntl), a                       ; READON_m | CRCON_m
	in	a, (data)                       ; read the AM
	ld	b, a
	in	a, (clock)
	cp	IBM_AM_CLK                      ; and the correct clock byte?
	jp	nz, wfs_id_am
	ld	a, b
	cp	IBM_ID_AM_DPAT
	jp	nz, wfs_id_am                   ; not the AM, try again
	in	a, (data)                       ; read the track number
	pop	de                              ; get the flags word
	ld	b, a
	cp	(iy + YB(target_cyl_hd))        ; reached target_cyl_hd?
	jr	z, .skipwt
	set	SIO_D_WRONG_TRACK, d
.skipwt:
        in	a, (data)
	cp	0                               ; should have a 00h byte here
	jr	z, .skipb5
	set	SIO_D_BIT5, d                   ; bad id field 9th byte not 00h
.skipb5:
        in	a, (data)                       ; sector number
	ld	c, a
	cp	(ix + DriveData.sector)
	jr	z, .skipws
	set	SIO_D_WRONG_SECTOR, d
.skipws:
        in	a, (data)
	cp	0                               ; another 00h byte here
	jr	z, .skipd4
	set	SIO_D_BIT4, d                   ; bad id field 11th byte not 00h
.skipd4:
        in	a, (data)                       ; first CRC byte
	set	SIO_D_SECTOR_FOUND, d
	call	verify_crc
	jp	wfs_coda


; verify CRC
; on entry:
;       The first CRC byte has been read?
;       IX: DriveData pointer
;       E: stat2?
;
; on exit:
;       E: drv_status | E with READY complemented

verify_crc:
        in	a, (data)                       ; process the second CRC byte
	ld	a, (ix + DriveData.drv)
	ld	h, a
	and	~(HEADLOAD_m | MGNENA_m)        ; turn off the IN-USE and error signals enable?
	out	(drv), a
	in	a, (data)                       ; one more to finalize the CRC check?
	ld	a, h
	and	~HEADLOAD_m                     ; turn on the error signals?
	out	(drv), a
;x0abb:
sio_done_get_status:
        in	a, (drv_status)
	ld	l, a
	xor	a
	out	(cntl), a                       ; turn off cntl signals
	ld	a, l                            ; A: drv_status value
	and	OVERUN_m | CRCERR_m | READY_m
	or	e                               ; Or in the other bits captured during the I/O
	bit	READY, a                        ; Toggle READY bit - not sure why this
	set	READY, a                        ; is done this way, do_sector_io_check could
	jr	z, .skip                        ; just reverse the sense of the bit test later.
	res	READY, a
.skip:	ld	e, a                            ; SIO_E_NOT_READY == ~READY
	ret

do_read_sector:
        ld	a, IU_READ
	set_in_use_leds
	call	wait_for_sector_id_field        ; DE: SIO_* flags
	ld	a, 1                            ; # retry attempts?
	jr	nz, do_read_sector_exit         ; failed

do_read_sector_common:
        push	hl
	ld	h, a                            ; H: # retry attempts?
	push	bc
	ld	a, CRCON_m
	out	(cntl), a
	set	SIO_E_3, e                      ; ?
	push	de                              ; save SIO_* flags from wait_for_sector_id_field
	ld	b, 0beh
	ld	d, HP_AM_CLK
	ld	c, switches + NOSW              ; switch port without switch bits
	ld	e, 7fh                          ; mask for HP_ID_AM_DPAT and HP_DEFECTIVE_TRACK_AM_DPAT
	ld	l, READON_m | CRCON_m
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jp	nz, do_read_sector_ibm

do_read_sector_hp:
        in      (c)                             ; undocumented Z80: in flags, (c)
	jp	p, do_read_sector_hp_retry      ; timedout
	in	a, (data)                       ; track number
	and	e                               ; mask AM for valid and defective AMs?
	ld	e, a
	in	a, (clock)
	cp	d                               ; have the correct AM clock pattern?
	jr	nz, do_read_sector_retry
	ld	a, l
	out	(cntl), a                       ; READON_m | CRCON_m - start CRC calcs
	in	a, (data)
	ld	d, a                            ; first byte of sector data
	ld	a, e                            ; test masked AM
	cp	HP_DATA_AM_DPAT
	jr	nz, do_read_sector_retry_hp_clk
	ld	hl, data_buffer+1               ; what are these shennanigans?
	ld	(hl), d                         ; sector data is byte swapped??
	dec	hl                              ; compatability with other systems?
	in	a, (data)
	ld	(hl), a                         ; second byte of sector data
	inc	hl
	inc	hl
	ld	c, data
	ld	b, 254
.readloop:                                      ; continue reading until 256 bytes saved
        inc	hl
	ind
	ini
	inc	hl
	jr	nz, .readloop

do_read_sector_verify_crc:
        in	a, (data)
	pop	de
	ld	(hl), 1                         ; ?
	call	verify_crc                      ; DE: SIO_* flags
	ld	a, b

do_read_sector_cleanup_exit:
        pop	bc
	pop	hl

do_read_sector_exit:
        call	show_previous_in_use_leds
	ret

do_read_sector_hp_retry:
        djnz	do_read_sector_hp
	dec	h
	jr	nz, do_read_sector_hp

do_read_sector_retries_exhausted:
        xor	a
	out	(cntl), a               ; stop the read attempts
	pop	de
	set	SIO_E_1, e              ; retries exhausted?
	inc	a                       ; set NZ?
	jr	do_read_sector_cleanup_exit

do_read_sector_retry_hp_clk:
        ld	d, HP_AM_CLK
do_read_sector_retry:
        ld	e, 7fh                  ; AM mask?
	xor	a
	out	(cntl), a               ; reset CRC chip
	ld	a, CRCON_m
	out	(cntl), a
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	z, do_read_sector_hp_retry
	jr	do_read_sector_ibm_retry

do_read_sector_ibm:
        in      (c)                             ; undocumented Z80: in flags, (c)
	jp	p, do_read_sector_ibm_retry     ; AMDT not asserted must be a TIMEOUT
	ld	a, l
	out	(cntl), a                       ; READON_m | CRCON_m
	in	a, (data)
	sub	IBM_DELETED_DATA_AM_DPAT
	jr	z, .doread
	cp	3                               ; IBM_DATA_AM_DPAT - IBM_DELETED_DATA_AM_DPAT = 3
	jr	nz, do_read_sector_retry
.doread:
        ld	d, a                            ; save the subtraction result
	in	a, (clock)
	cp	IBM_AM_CLK
	jr	nz, do_read_sector_retry
	ld	c, data                         ; data port number
	ld	hl, data_buffer
	ld	b, IBM_BYTES_PER_SECTOR
	inir                                    ; save the sector data
	ld	a, d                            ; A: the AM subtraction result
	pop	de                              ; SIO_* flags from wait_for_sector_id_field
	cp	3
	jr	z, .valid
	set	SIO_E_DEFECTIVE_SECTOR, e
.valid:	push	de
	ld	b, IBM_BYTES_PER_SECTOR
	jr	do_read_sector_verify_crc

do_read_sector_ibm_retry:
        djnz	do_read_sector_ibm
	jr	do_read_sector_retries_exhausted


cmd_unbuffered_write:
	call	sio_common_prep
	call	verify_write_enabled
	ld	b, 0
	call	wait_for_HPIB_SAD
	load_head
.again:	call	receive_sector_data
	call	head_unload_timer_1s_check
	ld	hl, 0f057h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | SIO_E_NOT_READY_m | SIO_E_DEFECTIVE_SECTOR_m | SIO_E_1_m | SIO_E_SCE_m)
	call	write_sector
	jp	nz, listen_terminate
	ld	a, (listen_transfer_final_d0d1)
	cp	IN_EOI
	jr	nz, .again
	call	normal_and_deselect_drives
	ret


cmd_buffered_write:
	call	sio_common_prep
	call	verify_write_enabled
	ld	b, 0
	call	wait_for_HPIB_SAD
	load_head
	call	receive_sector_data
	call	head_unload_timer_1s_check
	ld	hl, 0f057h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | SIO_E_NOT_READY_m | SIO_E_DEFECTIVE_SECTOR_m | SIO_E_1_m | SIO_E_SCE_m)
	call	write_sector
	call	z, normal_and_deselect_drives
	ret

write_sector:
        call	seek_to_target_cyl_hd
	jr	z, .ok
	jp	nc, abort_seek_check
	jp	c, abort_drive_fault_seek
.ok:	call	do_write_sector
	call	do_sector_io_check
	jr	nc, write_sector
	ret	nz
	call	next_logical_sector
	xor	a
	ret

do_write_sector:
        ld	a, IU_WRT
	set_in_use_leds
	call	set_lowcurr_and_precmp
	push	hl                              ; save error masks
	call	wait_for_sector_id_field
	jr	nz, write_sector_cleanup_ret
	ld	hl, data_buffer
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	nz, write_ibm_sector
	ld	b, HP_LOWCURR_START_TRACK
.wait:	djnz	.wait
	ld	a, WRITON_m
	out	(cntl), a
	xor	a
	out	(clock), a
	out	(data), a                       ; first 00h byte clock sync data field
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	xor	a
	out	(data), a                       ; second 00h byte clock sync
	ld	c, data
	out	(data), a                       ; third 00h byte clock sync
	out	(data), a                       ; fourth 00h byte clock sync
	ld	b, 4
	dec	a
.loop:	out	(data), a                       ; write 4 ffh byte clock sync
	djnz	.loop
	ld	a, 38h
	out	(clock), a
	ld	a, HP_DATA_AM_DPAT
	out	(data), a
	xor	a
	out	(clock), a
	ld	b, a
	ld	a, READON_m | WRITDRV_m | WRITON_m
	out	(cntl), a
.wrtloop:
        inc	hl                              ; sector data is byte swapped
	outd
	outi
	inc	hl
	jr	nz, .wrtloop
write_sector_coda:
        ld	a, READON_m | CRCOUT_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	xor	a
	out	(data), a                       ; CRC bytes get written here
	out	(data), a
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	a, b
	out	(data), a                       ; ?
	out	(data), a
	call	sio_done_get_status

write_sector_cleanup_ret:
        xor	a
	out	(cntl), a
	call	show_previous_in_use_leds
	pop	hl
	ret

;x0c4c:
write_ibm_sector:
        ld	b, 71                           ; delay a bit
.delay:	djnz	.delay
	ld	a, 4
	out	(cntl), a
	ld	b, 5
	ld	a, 0ffh
	out	(clock), a
	xor	a
	out	(data), a
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	xor	a
.loop:	out	(data), a
	djnz	.loop
	ld	a, READON_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	a, IBM_AM_CLK
	out	(clock), a
	ld	a, IBM_DATA_AM_DPAT
	bit	DBIT_ON, (iy + YB(state_flags))
	jr	z, .notdel
	ld	a, IBM_DELETED_DATA_AM_DPAT
.notdel:
        out	(data), a
	ld	a, 0ffh
	out	(clock), a
	ld	b, IBM_BYTES_PER_SECTOR
	ld	c, data
	otir
	dec	b
	jr	write_sector_coda

head_unload_timer_1s_check:
        ld	d, 0ffh                  ; approx 1 s elapsed?
	call	head_unload_timer_check
	ret


	fillto	0ca0h, 076h


cmd_initialize:
	call	check_for_holdoff
	call	verify_unit_ready
	call	verify_known_format
	call	verify_cylinder_jv
	call	verify_write_enabled
	load_head
	seek
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	nz, .initibm
	ld	d, 0
	ld	(iy + YB(format_flags)), d      ; reset the format_flags
	ld	e, HP_ID_AM_DPAT
	bit	INIT_DBIT, (iy + YB(cmd_buffer))
	jr	z, .skip
	ld	e, HP_DEFECTIVE_TRACK_AM_DPAT   ; D bit was set in the command
.skip:	ld	c, (ix + DriveData.target_cyl_hd)
	ld	b, 2                            ; Initialize interleave value
	call	fmt_hp_track
	jp	nz, abort_drive_fault
	jr	.init_common

.initibm:
        bit	INIT_DBIT, (iy + YB(cmd_buffer))
	jr	z, .init_common
	set	DBIT_ON, (iy + YB(state_flags))

.init_common:
        ld	b, 0
	call	wait_for_HPIB_SAD
	call	receive_sector_data
	ld	hl, 0f053h
        ; (SIO_D_WRONG_SECTOR_m | SIO_D_WRONG_TRACK_m | SIO_WRONG_HEAD_m | SIO_BIT5_m) << 8
        ; | (OVERUN_m | SIO_E_NOT_READY_m | SIO_E_1_m | SIO_E_SCE_m)
	call	write_sector_jv
	res	DBIT_ON, (iy + YB(state_flags))
	call	z, normal_and_deselect_drives
	ld	a, (S1)
	cp	S1_DBIT_m | S1_DTE
	call	z, normal_and_deselect_drives
	ret


; cmd_format - format a disc
cmd_format:
	call	check_for_holdoff
	call	verify_unit_ready
	call	verify_write_enabled
	ld	a, (format_type)
	and	FORMAT_WANTED_TYPE_M
	cp	FORMAT_TYPE_HP
	jr	z, .typegood
	cp	FORMAT_TYPE_IBM
	jp	nz, abort_IO_program_error
.typegood:
        rlca                                    ; convert into the desired DISC_TYPE value
	ld	b, a
	ld	a, (format_interleave)          ; validate the interleave value
	cp	1
	jp	m, abort_IO_program_error
	ld	c, HP_SECTORS_PER_TRACK
	bit	DT_IBM, b
	jr	z, .valid
	ld	c, IBM_SECTORS_PER_TRACK
.valid: cp	c
	jp	nc, abort_IO_program_error
	ld	a, (ix + DriveData.stat2)
	ld	(previous_format), a
	and	DT_DS_m
	or	b                               ; add in the double sided flag
	bit	DT_IBM, a
	jr	z, .savestat2
	bit	DT_DS, a
	jp	nz, abort_stat2_error           ; IBM format is single sided only
.savestat2:
        ld	(ix + DriveData.stat2), a

	ld	b, NUNITS                       ; unload the other drive's heads
.uloop:
        dec	b                               ; as this will take a while
	jp	m, format_disc
	ld	a, b
	cp	(iy + YB(current_unit))
	call	nz, maybe_unload_head
	jr	.uloop

format_disc:
        call	deselect_drives
	ld	a, (current_unit)
	call	prep_drive_regs
	ld	a, IU_FMT
	set_in_use_leds
	load_head
	reset_drive
	jp	nz, abort_drive_fault_seek
	ld	bc, 0
	ld	(iy + YB(format_flags)), b      ; reset the format_flags
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jp	nz, do_ibm_format
.hpfmtloop:
        ld	a, b
	cp	PHYSICAL_TRACKS
	jr	z, format_done
	push	bc
	bit	FORMAT_OVERRIDE, (iy + YB(format_type))
	jr	nz, .hpdofmt
	bit	DT_HP, (iy + YB(previous_format))
	jr	z, .hpdofmt                     ; blank media or different format
	wait_for_sector                         ; no, check current format?
	cp	0ffh
	jr	z, .x0d85
	bit	2, e                            ; DT_HP?
	jr	z, .hpdofmt
.x0d85:	ld	c, HP_DEFECTIVE_TRACK_MARKER
	call	get_interleave_and_data_byte
	call	fmt_hp_track
	jp	nz, abort_drive_fault
	pop	bc
	jr	.x0dae

.hpdofmt:
        pop	bc
	push	bc                              ; b = current physical track or logical track?
	call	get_interleave_and_data_byte
	call	fmt_hp_track
	jp	nz, abort_drive_fault
	pop	bc
	bit	DT_DS, (ix + DriveData.stat2)
	jr	z, .skipss1
	bit	7, c                            ; what is C here?
	set	7, c
	jr	z, .x0dae
	res	7, c
.skipss1:
        inc	c
.x0dae:	bit	DT_DS, (ix + DriveData.stat2)
	jr	z, .skipss2
	bit	HEAD1, (ix + DriveData.drv)     ; remember which head was selected
	set	HEAD1, (ix + DriveData.drv)     ; select head 1
	jr	z, .skipinc                     ; head 0 was selected previously
	res	HEAD1, (ix + DriveData.drv)     ; select head 0
.skipss2:
        inc	b
.skipinc:
        deassert_in_use
	bit	HEAD1, (ix + DriveData.drv)
	call	z, step_in_one_track            ; head 0 was selected
	xor	a
	bit	HEAD1, (ix + DriveData.drv)
	ld	a, (fmt_data_1)
	jr	nz, .x0dd8
	ld	a, (fmt_data_2)
.x0dd8:	add	a, (iy + YB(format_phys_sector))
	cp	HP_SECTORS_PER_TRACK
	jr	c, .x0de1
	sub	HP_SECTORS_PER_TRACK
.x0de1:	ld	(format_phys_sector), a
	jr	.hpfmtloop

format_done:
        reset_drive
	jp	nz, abort_drive_fault_seek
	call	normal_and_deselect_drives
	ret

; on exit:
;       B: interleave
;       D: data byte
get_interleave_and_data_byte:
        ld	b, (iy + YB(format_interleave))
	ld	d, (iy + YB(format_data_byte))
	ld	e, 70h  ; HP_ID_AM_DPAT?
	ret

;x0df7:
do_ibm_format:
        push	bc
	bit	FORMAT_OVERRIDE, (iy + YB(format_type))
	jr	nz, .skip
	bit	DT_IBM, (iy + YB(previous_format))
	jr	nz, .previous_type_is_ibm
.skip:	pop	bc
	push	bc
	call	get_interleave_and_data_byte
	call	fmt_ibm_track_jv
	jp	nz, abort_drive_fault
	pop	bc
	inc	c
.ibmfmtloop:
        inc	b
	ld	a, b
	cp	PHYSICAL_TRACKS
	jr	z, format_done
	call	step_in_one_track
	jr	do_ibm_format

.previous_type_is_ibm:
        ld	hl, 0064h                       ; HL = error masks for wfs_coda?
	res	MGNENA, (ix + DriveData.drv)
.x0e23:	call	read_sector_jv
	res	FF_DATA_INIT, (iy + YB(format_flags))
	bit	OVERUN, e
	jr	nz, .x0e45
	bit	SIO_D_SECTOR_FOUND, d
	jr	z, .ibmcoda
	ld	a, b
	cp	0ffh
	jr	z, .ibmcoda
	bit	SIO_E_1, e
	jr	nz, .x0e45
	bit	CRCERR, e
	jr	nz, .x0e45
	bit	SIO_E_DEFECTIVE_SECTOR, e
	jr	nz, .ibmcoda
	jr	.ibmfmtloop

.x0e45:	bit	SECTOR_IO_RETRY, (iy + YB(state_flags))
	set	SECTOR_IO_RETRY, (iy + YB(state_flags))
	jr	z, .ibmcoda
	res	SECTOR_IO_RETRY, (iy + YB(state_flags))
.ibmcoda:
        ld	c, 0ffh
	call	get_interleave_and_data_byte
	call	fmt_ibm_track_jv
	jp	nz, abort_drive_fault
	pop	bc
	jr	.ibmfmtloop

step_in_one_track:	push	bc
	ld	c, 0                            ; step in towards center
	call	step_drive_jv
	ld	b, 20
	call	delay                           ; delay 20 ms
	pop	bc
	inc	(ix + DriveData.physical_cylinder)
	ret

; do_hp_fmt_track - format an HP type track
;
; on entry
;       B: interleave
;       C: target_cyl_hd
;       D: data byte
;       E: ID AM
;
; on exit:
;       A = 0 (success)
;       A = 1 (failure)

do_hp_fmt_track:
        bit	FF_DATA_INIT, (iy + YB(format_flags))
	jr	nz, .skipinit
	push	de
	push	bc
	call	init_sector_table
	pop	bc
	ld	hl, format_data_tbl
	ld	a, b
	dec	a
	rlca
	ld	e, a
	add	hl, de                  ; HL = &fmt_data_tbl[2 * (interleave - 1)]
	ld	d, (hl)
	inc	hl
	ld	e, (hl)
	ld	(fmt_data_2), de
	xor	a
	ld	(format_phys_sector), a
	pop	de
	call	init_hp_format_sector_buffer
	set	FF_DATA_INIT, (iy + YB(format_flags))

;x0e98:
.skipinit:
        ld	b, 0
	ld	a, c                            ; C: target_cyl_hd
	cp	HP_DEFECTIVE_TRACK_MARKER
	jr	z, .skip
	ld	b, a
	and	BYTENOT(HEAD1_m)                ; just the cylinder

.skip:  push	af
	ld	(iy + YB(target_cyl_hd)), b
	call	set_lowcurr_and_precmp
	ld	(iy + YB(fmt_sector_count)), HP_SECTORS_PER_TRACK
	ld	c, data                         ; data port address
	call	get_log_sector_ptr              ; DE = logical sector number pointer
	pop	af                              ; A = cylinder
	push	ix                              ; save DriveData pointer
	ld	ix, format_sector_buffer + 9
	ld	(ix + 0), a                     ; stuff the cylinder number into the buffer
	ld	b, 0
	xor	a
	out	(clock), a
;x0ec1:
.widx1:
        in	a, (drv_status)                 ; wait for the index hole signal signal
	bit	READY, a
	jp	z, fmt_track_error
	bit	INDEX, a
	jr	z, .widx1

	ld	a, r_OVERUN_m                   ; prepare to format the track
	out	(reset), a
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a

.widxd1:                                        ; while waiting for the index signal
        out	(c), b                          ; to be deasserted, write zeros
	in	a, (drv_status)                 ; erase end of gap 3?
	bit	READY, a
	jp	z, fmt_track_error
	bit	INDEX, a
	jr	nz, .widxd1

.widx2:	out	(c), b                          ; wait for the index signal again (erase track)
	in	a, (drv_status)
	bit	READY, a
	jp	z, fmt_track_error
	bit	INDEX, a
	jr	z, .widx2

;x0eee
.widxd2:
        out	(c), b                          ; finish erase track waiting for start of track
	in	a, (drv_status)
	bit	INDEX, a
	jr	nz, .widxd2

; NOTE: Various bits of processing are done in between writes of floppy write data bytes
;       to maintain timing.
;x0ef6
.fmt_sector:
        ld	hl, format_sector_buffer
	outi                                    ; first zero byte of ID field clock sync
	ld	a, (de)                         ; A = logical sector number
	bit	HEAD1, (iy + YB(target_cyl_hd)) ; add in the head bit
	jr	z, .current_cyl_hd0
	set	HEAD1, a
.current_cyl_hd0:
        outi                                    ; second zero byte of ID field clock sync
	ld	b, 6
	ld	(ix + 1), a                     ; stuff the sector number into the buffer
	otir                                    ; write 6 bytes (completes clock sync)
	ld	a, 38h
	out	(clock), a
	outi                                    ; write the AM with the required clock pattern
	xor	a
	out	(clock), a
	ld	a, READON_m | WRITDRV_m | WRITON_M
	out	(cntl), a
	outi                                    ; write track number
	outi                                    ; write sector number
	ld	a, READON_m | CRCOUT_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	outi                                    ; write ID field CRC byte 1
	outi                                    ; write ID field CRC byte 2
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	b, 25
	otir                                    ; write 25 bytes gap 1 + data field sync
	ld	a, 38h
	out	(clock), a
	outi                                    ; write the data field AM with the required clock
	xor	a
	out	(clock), a
	ld	b, a
	ld	a, READON_m | WRITDRV_m | WRITON_M
	out	(cntl), a
	otir                                    ; write sector data fill 256 bytes
	ld	a, READON_m | CRCOUT_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	outi                                    ; write data field CRC byte 1
	outi                                    ; write data field CRC byte 2
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	b, 35
	outi                                    ; write 35 bytes (1 zero byte + gap 2)
	ld	a, (format_phys_sector)
	inc	a
        ; is gap2 actually larger than documented? gap 3 smaller?
	outi                                    ; write one byte - what is this?? B = 255
	cp	HP_SECTORS_PER_TRACK
	jr	c, .skipwrap                    ; format_phys_sector < HP_SECTORS_PER_TRACK
	xor	a                               ; sector numbers wrapped around
.skipwrap:
        ld	(format_phys_sector), a
	outi                                    ; write one byte - what is this?? B = 254
	ld	de, sector_table                ; get address of the next logical sector number
	add	a, e
	ld	e, a
	outi                                    ; write one byte - what is this?? B = 253
	dec	(iy + YB(fmt_sector_count))
	jr	z, fmt_track_done
	otir                                    ; write 253 bytes gap 3?
	jr	.fmt_sector

;x0f6e:
fmt_track_done:	xor	a
	out	(cntl), a               ; turn off signals
	in	a, (drv_status)
	bit	OVERUN, a
	jr	nz, fmt_track_error
	ld	b, 152                  ; wait a bit (for what?)
.loop:	djnz	.loop
	xor	a                       ; indicate success
	pop	ix                      ; restore the DriveData pointer
	ret

;x0f7f:
fmt_track_error:
        xor	a
	out	(cntl), a               ; turn off enable signals
	pop	ix                      ; restore the DriveData pointer
	or	1                       ; indicate failure
	ret

; get_log_sector_ptr - lookup the logical sector number for this physical sector number
;
; on exit:
;       DE = address of the logical sector number

;x0f87:
get_log_sector_ptr:
        ld	de, sector_table
	ld	a, (format_phys_sector)
	add	a, e
	ld	e, a
	ret

        ; 29 two byte entries correspond to the sector interleave in format_track
        ; 2nd byte of entry loaded into fmt_data_2, 1st byte into fmt_data_1

format_data_tbl:
        db      28, 24  ; 1
        db      28, 24  ; 2
        db      28, 24  ; 3
        db      29, 26  ; 4
        db      26, 24  ; 5
        db      25, 26  ; 6
        db       0,  0  ; 7
        db      29, 29  ; 8
        db      28, 28  ; 9
        db      21, 21  ; 10
        db       0,  0  ; 11
        db      25, 25  ; 12
        db       0,  0  ; 13
        db      29, 29  ; 14
        db      16, 16  ; 15
        db      29, 29  ; 16
        db       0,  0  ; 17
        db      25, 25  ; 18
        db       0,  0  ; 19
        db      21, 21  ; 20
        db      28, 28  ; 21
        db      29, 29  ; 22
        db       0,  0  ; 23
        db      25, 25  ; 24
        db      26, 26  ; 25
        db      29, 29  ; 26
        db      28, 28  ; 27
        db      29, 29  ; 28
        db       0,  0  ; 29

; init_hp_format_sector_buffer - setup the sector buffer with the HP format bytes

init_hp_format_sector_buffer:
        ld	hl, format_sector_buffer
	xor	a
	ld	b, 4
	call	fill_buffer_with_a      ; 4 bytes of 00h (ID field clock sync)
	dec	a
	ld	b, 4
	call	fill_buffer_with_a      ; 4 bytes of ffh (ID field clock sync)
	ld	(hl), e                 ; ID AM
	inc	hl
	xor	a
	ld	b, 25
	call	fill_buffer_with_a      ; 25 bytes of 00h (..., Gap 1, data field clock sync)
	dec	a
	ld	b, 4
	call	fill_buffer_with_a      ; 4 bytes of ffh (data field clock sync)
	ld	(hl), HP_DATA_AM_DPAT
	inc	hl
	ld	a, d
	call	fill_buffer_with_a      ; 256 bytes of fill byte (B was zero)
	xor	a
	ld	b, 37
	call	fill_buffer_with_a      ; 37 bytes of 00h (CRC, 00, Gap 2)
	ret

; fill_buffer_with_a - fills a buffer with the byte in A
;
; on entry:
;       A: the fill byte
;       B: the number of fill bytes
;       HL: the buffer pointer

fill_buffer_with_a:
        ld	(hl), a
	inc	hl
	djnz	fill_buffer_with_a
	ret

; init_sector_table - fill the table of the logical sector numbers at the
;                      specified interleave
;
; on entry
;       B: interleave

init_sector_table:
        ld	hl, sector_table
	push	hl
	ld	(iy + YB(format_flags)), b              ; format_flags temporarily is interleave
	ld	b, HP_SECTORS_PER_TRACK
	ld	a, 0ffh                                 ; sector number not yet set
	call	fill_buffer_with_a                      ; 30 bytes of 0ffh, B = 0 after
	pop	hl
	ld	c, HP_SECTORS_PER_TRACK                 ; C: sectors per track
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	z, .skiphp
	inc	b                                       ; B: first sector number
	ld	c, IBM_SECTORS_PER_TRACK
.skiphp:
        ld	de, 0                                   ; E: current physical sector
	ld	(hl), b                                 ; first sector of the track
	inc	b
.loop:	push	hl
	ld	a, e
	add	a, (iy + YB(format_flags))              ; E += interleave
.chkwrap:
        cp	c
	jr	c, .nowrap
	sub	c                                       ; E -= sectors_per_track
.nowrap:
        ld	e, a
	add	hl, de                                  ; address of this sector in the table
	ld	a, (hl)
	cp	0ffh
	jr	z, .claim                               ; not yet assigned a sector number
	inc	e                                       ; bump to an unclaimed phys sector
	pop	hl
	push	hl
	ld	a, e
	jr	.chkwrap

.claim:	ld	(hl), b                                 ; set this logical sector number in table
	inc	b
	ld	a, b
	cp	c                                       ; finished?
	pop	hl
	jr	nz, .loop                               ; no
	bit	FORMAT_IBM, (iy + YB(disc_format))
	ret	z                                       ; HP format
	cp	IBM_SECTORS_PER_TRACK + 1
	ret	z
	inc	c                                       ; one more time! Index AM??
	jr	.loop

; format IBM track
; B: interleave
; C: ?
; D: format data byte
; E: ?
fmt_ibm_track:
        xor	a
	ld	(format_phys_sector), a
	bit	FF_DATA_INIT, (iy + YB(format_flags))
	jr	nz, .skipsetup
	push	bc
	push	de
	call	init_sector_table
	pop	de
	call	fill_ibm_sector_buffer
	set	FF_DATA_INIT, (iy + YB(format_flags))
	pop	bc
.skipsetup:
        ld	a, IBM_SECTORS_PER_TRACK
	ld	(fmt_sector_count), a
	ld	b, c
	push	bc
x1060:	ld	b, 2eh  ; 46
	ld	c, data
	call	get_log_sector_ptr              ; DE = logical sector number pointer
	ld	hl, format_sector_buffer
	pop	af                              ; AF = BC?
	push	ix
	ld	ix, format_sector_buffer
	ld	(ix + 0), a
	cp	0ffh
	jr	nz, .x107e
	ld	(ix + 1), a
	ld	(ix + 3), a
.x107e:	ld	a, 0ffh
	out	(clock), a
.waitidx:
        in	a, (drv_status)                 ; wait for the index hole asserted
	bit	READY, a
	jp	z, fmt_track_error
	bit	INDEX, a
	jr	nz, .waitidx

.waitd:	in	a, (drv_status)                 ; wait for index hole deasserted
	bit	READY, a
	jp	z, fmt_track_error
	bit	INDEX, a
	jr	z, .waitd

	ld	a, r_OVERUN_m
	out	(reset), a
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	otir
	ld	a, IBM_INDEX_AM_CLK
	out	(clock), a
	outi
	ld	a, 0ffh
	out	(clock), a
	ld	b, 26                           ; gap 0
	otir
x10b0:	ld	b, 6                            ; 6 zero bytes clock sync
	outi
	ld	a, (ix + 0)
	cp	0ffh
	jr	z, x10bc                        ; deleted track?
	ld	a, (de)
x10bc:	ld	(ix + 2), a
	otir
	ld	a, READON_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	a, IBM_AM_CLK
	out	(clock), a
	outi
	ld	a, 0ffh
	out	(clock), a
	ld	b, 4
	otir
	ld	a, READON_m | CRCOUT_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	outi                                    ; write id field CRC byte 1
	outi                                    ; write id field CRC byte 2
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	b, 17                           ; gap 1 + 6 zero bytes clock sync
	otir
	ld	a, READON_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	a, IBM_AM_CLK
	out	(clock), a
	outi
	ld	a, 0ffh
	out	(clock), a
	ld	b, 128                          ; sector fill bytes
	otir
	ld	a, READON_M | CRCON_m | WRITDRV_m | WRITON_m
	out	(cntl), a
	outi                                    ; write data field CRC byte 1
	outi                                    ; write data field CRC byte 2
	ld	a, WRITDRV_m | WRITON_m
	out	(cntl), a
	ld	b, 26                           ; gap 2
	outi
	ld	a, (format_phys_sector)
	inc	a
	ld	de, data_buffer_count
	outi
	cp	1ah
	jr	c, x1113
	xor	a
x1113:	ld	(format_phys_sector), a
	add	a, e
	ld	e, a
	otir
	ld	hl, x60b4
	dec	(iy + YB(fmt_sector_count))
	jr	nz, x10b0                       ; format next sector
	dec	b
x1123:	out	(c), b
	in	a, (drv_status)
	bit	4, a
	jp	z, fmt_track_error
	bit	0, a
	jr	z, x1123
	out	(c), b
	jp	fmt_track_done

fill_ibm_sector_buffer:
        ld	hl, format_sector_buffer
	ld	a, IBM_GAP_DPAT
	ld	b, 40
	call	fill_buffer_with_a              ; last 40 bytes of gap 3 bytes
	xor	a
	ld	b, 6
	call	fill_buffer_with_a              ; 6 bytes of 00h sync bytes
	ld	(hl), IBM_INDEX_AM_DPAT
	inc	hl
	dec	a
	ld	b, 26
	call	fill_buffer_with_a              ; 26 bytes of 0ffh gap 0
	xor	a
	ld	b, 6
	call	fill_buffer_with_a              ; 6 bytes of 00h sync bytes
	ld	(hl), IBM_ID_AM_DPAT            ; id field ID AM
	inc	hl
	ld	b, 6
	call	fill_buffer_with_a              ; 6 bytes to be filled in later
	dec	a
	ld	b, 11
	call	fill_buffer_with_a              ; 11 bytes of 0ffh gap 1
	xor	a
	ld	b, 6
	call	fill_buffer_with_a              ; 6 bytes of 00h sync bytes
	ld	(hl), IBM_DATA_AM_DPAT          ; data field AM
	inc	hl
	ld	a, d
	ld	b, 128
	call	fill_buffer_with_a              ; 128 bytes of sector data
	xor	a
	ld	b, 2
	call	fill_buffer_with_a              ; space for CRC bytes
	dec	a
	ld	b, 26
	call	fill_buffer_with_a              ; 26 bytes of 0ffh gap 2
	ret

	halt
	halt


; Drive has D and DL jumpers closed:
;       In Use = active + Drive Select activated latches door lock on
;       In Use = inactive + Drive Select activated unlatches door lock

cmd_door_lock:
	call	check_for_holdoff
	call	check_unit_ready
	call	load_head_ix
	ld	a, (ix + DriveData.xv)          ; remove this drive from the head_unload_set
	cpl                                     ; form mask
	and	(iy + YB(head_unload_set))
	ld	(head_unload_set), a
	set	DOOR_LOCK, (ix + DriveData.drv)
	jr	door_lock_coda

cmd_door_unlock:
	call	check_for_holdoff
	call	check_unit_ready
	res	DOOR_LOCK, (ix + DriveData.drv)
	ld	b, (iy + YB(current_unit))
	call	maybe_unload_head
door_lock_coda:
        call	normal_and_deselect_drives
	ret


        ; return 2 bytes cylinder, 1 byte head, 1 byte sector
cmd_request_logical_address:
	ld	a, (current_unit)
	call	prep_drive_regs
	ld	b, (ix + DriveData.target_cyl_hd)      ; B = second byte
	bit	HEAD1, b                        ; sets Z (head) appropriately for later
	res	HEAD1, b                        ; clear the head from the rest of the cylinder
	ld	d, (ix + DriveData.sector)      ; D = fourth byte
	jr	cmd_request_address_common

        ; return 2 bytes cylinder, 1 byte head, 1 byte zeros
cmd_request_physical_address:
	ld	a, (current_unit)
	call	prep_drive_regs
	ld	b, (ix + DriveData.physical_cylinder)
	ld	d, 0                            ; D = fourth byte
	bit	HEAD1, (ix + DriveData.drv)

cmd_request_address_common:
        ld	e, 0                            ; E = third byte
	jr	z, .skip                        ; head 0
	inc	e                               ; head 1
.skip:	ld	c, 0                            ; C = first byte (cylinder msb - always zero)
	call	check_for_holdoff
	jp	send_status_or_address_jv


        ; send the contents of the data buffer (256 bytes max)
        ; Note: the controller can terminate this early by UNT or a SAD is sent
read_loopback:
        ld	a, 0ffh                         ; 255 bytes + 1 final byte with EOI
read_loopback_talk:
        ld	hl, data_buffer_count
	ld	b, FLUSH_FIFO_m | SEND_EOI_m
	call	do_talk_transfer
	xor	a
	ld	(S1), a                         ; S1_NORMAL
	ret

        ; place up to 256 bytes into the data buffer
        ; Must mark last byte with EOI set if less than 256 bytes is sent.
write_loopback:
        ld	hl, data_buffer_count
	xor	a                               ; A = 0 -> up to 256 bytes
	call	do_listen_transfer
	ret

        ; download up to 256 bytes at data_buffer
        ; jumps to download_code_entry but only if the download was terminated with EOI
download:
        call	write_loopback
	ld	a, (listen_transfer_final_d0d1)
	cp	IN_EOI                           ; terminated with EOI marker?
	jp	nz, abort_IO_program_error      ; No 
	ld	hl, download_code_entry
	ex	de, hl
	call	prep_for_next_cmd               ; reset SP to ramend
	ex	de, hl
	jp	(hl)                            ; execute the downloaded code
                                                ; what happens if it returns to the caller?

initiate_self_test:
        ld	hl, data_buffer_count
	xor	a
	call	do_listen_transfer
	ld	a, (data_buffer_count)
	cp	1
	ld	d, (iy + YB(data_buffer))       ; data_buffer (cylinder)
	ld	e, 0                            ; default W to 0
	jr	z, .skip                        ; only received one byte
	ld	e, (iy + YB(data_buffer+1))     ; data_buffer+1 (W - seems to be bit 3 of second byte)
.skip:	ld	a, d
	cp	PHYSICAL_TRACKS
	jp	nc, abort_seek_check                ; cylinder in range?
	jp	do_initiate_self_test   ; yes


read_self_test:
        ld	hl, (leds_shdw)
	ld	a, h
	ld	h, l
	ld	l, a
	ld	(data_buffer), hl
	ld	a, 1
	jr	read_loopback_talk


	fillto	01240h, 076h


;******************************************************************
; delay - delays for (B) ms
;
; on entry:
;       B: the number of milliseconds to delay

delay:	push	bc
	ld	b, 0
.loop1:	djnz	.loop1
	ld	b, 28h
.loop2:	djnz	.loop2
	pop	bc
	djnz	delay
	ret


;******************************************************************
; prep_for_next_cmd - reset the execution state
;
; On exit:
;       SP = ramend
;       IY = ramstart
;       IX = &drive_data_table[0]
;       state_flags reset

prep_for_next_cmd:
        pop	hl                      ; save the return address
	xor	a
	ld	(state_flags), a        ; clear the state_flags
	ld	sp, ramend
	ld	iy, in_use_leds_stack 
	ld	(in_use_leds_stack_ptr), iy
	ld	iy, ramstart
	ld	ix, drive_data_table
	jp	(hl)                    ; jump to the return address


set_parity_check_state:
        in	a, (phi_int_mask)
	ld	b, a
	ld	a, (parity_check_state)
	out	(phi_status), a
	ld	a, b
	out	(phi_int_mask), a
	ret


do_PPD:	in	a, (phi_control)
	and	~RSPD_PP
	jr	write_phi_control

do_PPE:	in	a, (phi_control)
	or	RSPD_PP
	jr	write_phi_control

flush_out_fifo:
        in	a, (phi_control)
	or	INIT_FIFO

write_phi_control:
        out	(phi_control), a
	ret


; poll_input_or_DCL - check if a byte has been received or a device clear happened
;
; on exit:
;       NZ: a byte is available to read

        ; does the PHI clear the inbound fifo bytes if DCL received? No
poll_input_or_DCL:
        in	a, (phi_int_cond)
	bit	FIFO_BYTE, a
	ret	nz                              ; return if inbound fifo bytes are available
	bit	DEV_CLR, a
	call	nz, check_for_command_byte_jv
	ret

check_for_parity_error:
        in	a, (phi_int_cond)               ; get D0D1 bits associated with phi_int_cond
	in	a, (phi_status)
	bit	PRTY_ERR, a
	ret	z                               ; no
	ld	b, 40h                          ; ? never used by callers
	out	(phi_status), a                 ; D0D1=01 (clear parity error)
	xor	a
	out	(phi_int_cond), a
	ld	(iy + YB(DSJ)), DSJ_Parity_error
	ret

        ; B = the expected SAD (bit 5: TALK)
wait_for_HPIB_SAD:
        push	bc
	ld	a, IU_RECV
	call	show_in_use_leds_jv
	call	do_PPE
	ld	bc, 0                           ; waits for 41 * 65536 iterations
	ld	d, 41
.again:	call	poll_input_or_DCL
	jr	nz, .gotone                     ; got a byte in the inbound FIFO
	djnz	.again
	dec	c
	jr	nz, .again
	dec	d
	jr	nz, .again
	jp	abort_IO_program_error          ; timedout without having received the SAD

.gotone:
        call	do_PPD
	in	a, (phi_fifo)
	ld	c, a
	in	a, (phi_status)	                ; check D0D1 bits
	bit	6, a                            ; secondary address received?
	jp	z, abort_IO_program_error       ; no
	call	check_for_parity_error
	ld	a, c
	pop	bc
	cp	b                               ; does it match the expected secondary?
	jp	nz, abort_IO_program_error
	jp	show_previous_in_use_leds       ; returns to caller?

        ; A = byte, B = D0D1 bits
send_out_byte:
        push	af
	ld	a, DATA_FRZ_m
	out	(phi_status), a
	call	poll_input_or_DCL
	jp	nz, listen_terminate
	ld	a, b
	out	(phi_status), a          ; set D0D1 bits from caller
	pop	af
	out	(phi_fifo), a            ; send the byte
	ret

terminate_talk:
        ld	a, 1                     ; dummy byte
	ld	b, OUT_END
	push	af
	call	flush_out_fifo
	pop	af
	jp	send_out_byte

;================= START HERE ========================
        ; A = number of bytes, B = bit 1: flush out fifo, bit 0: EOI
        ; HL = address of cmd_count (can this potentially be anywhere??)
        ; HL+1 = address of the transfer buffer
do_talk_transfer:
        ld	(hl), a
	ld	a, IU_SEND
	call	show_in_use_leds_jv
	push	bc                      ; save the flags for later
	bit	FLUSH_FIFO, b
	call	nz, flush_out_fifo
	ld	a, DATA_FRZ
	out	(phi_status), a         ; unfreeze the outbound fifo
	in	a, (phi_int_cond)
	bit	FIFO_BYTE, a
	jr	nz, talkdone2          ; the inbound fifo has bytes
	set	TALK_ACT, (iy + YB(state_flags))
	call	enable_proc_abrt_int
	xor	a                       ; D0D1 = 00 (normal bytes)
	out	(phi_status), a
	ld	b, (hl)                 ; B = transfer count
	inc	hl                      ; the buffer follows the transfer count
.dotalk:
        ld	c, phi_fifo             ; phi_fifo port number
	otir                            ; send the data
	pop	bc                      ; get the flags back
	bit	SEND_EOI, b
	jr	z, .talkdone            ; don't send final EOI marked byte
	ld	b, 0                    ; clear the flags so we don't infinite loop!
	push	bc
	ld	b, 1                    ; transfer the final (dummy?) byte
	ld	a, OUT_END_m            ; D0D1 = 10 (set EOI)
	out	(phi_status), a
	jr	.dotalk

;x132e:
.talkdone:
        push	bc
talkdone2:
        res	TALK_ACT, (iy + YB(state_flags))
	pop	bc
	call	set_phi_int_mask
	call	show_previous_in_use_leds
	ret

talkdone3:
        pop	bc
	call	set_phi_int_mask
	call	check_for_command_byte_jv
	jr	talkdone2

        ; Preserves BC, DE, HL?
do_listen_transfer:
        push	bc
	push	de
	push	hl
	push	af                      ; A = max byte count (0 -> 256 bytes)
	ld	a, IU_RECV
	call	show_in_use_leds_jv
	inc	hl                      ; skip past read count?
	pop	bc                      ; pops A (byte count) into B for inir
	set	LISTEN_ACT, (iy + YB(state_flags))
restart_listen_transfer:
        call	enable_proc_abrt_int
	ld	c, phi_fifo
	xor	a                       ; not sure what this is for
	ei                              ; enable interrupts (PHI interrupts if transfer stops early?)
	inir                            ; read data from talker

        ; PHI IO Interrupt Handler
        ; On entry A = phi_status register value
        ; is there always an interrupt or just if it stops early?
terminate_listen_tranfer:
        di
	res	LISTEN_ACT, (iy + YB(state_flags))
	ld	c, a                    ; Save phi_status value
	call	set_parity_check_state
	pop	de                      ; pops saved HL value from do_listen_transfer
	push	hl
	push	de
	xor	a                       ; clear A (clear carry flag)
	sbc	hl, de                  ; HL = # bytes read
	ld	b, l
	dec	b                       ; B = adjusted bytes read
	pop	hl                      ; swaps pushed DE & HL, HL now data_buffer_count ptr
	pop	de                      ; DE now ptr to 1 past end of read bytes?
	ld	(hl), b                 ; store # bytes read
	ld	a, c                    ; Fetch saved phi_status value
	and	IN_EOI                  ; Isolate saved PHI D0 and D1 bits
	ld	(listen_transfer_final_d0d1), a ; Save them
	cp	SAD_D1_m
	call	set_phi_int_mask        ; restore idle phi_int_mask?
	jr	nz, .done               ; no
	dec	de                      ; adjust to the last byte received (a SAD byte)
	ld	a, (de)
	cp	HP300_CLEAR_SAD
	jp	z, HP_300_Clear_jv      ; listen was terminated by receipt of HP300 clear SAD
.done:	call	show_previous_in_use_leds
	pop	de
	pop	bc
	ret

; hmm - restarts the trasnfer but with the state_flag cleared??
x138a:	res	LISTEN_ACT, (iy + YB(state_flags))
	call	set_phi_int_mask        ; restore idle phi_int_mask?
	call	check_for_command_byte_jv
	dec	hl
	inc	b
	jr	restart_listen_transfer

abort_listen:
        res	LISTEN_ACT, (iy + YB(state_flags))
	call	set_phi_int_mask
	jp	abort_IO_program_error

; set_idle_phi_int_mask?
set_phi_int_mask:
        ld	a, (parity_check_state)         ; set PHI D0D1 for following phi_int_mask write
	out	(phi_status), a
	ld	a, FIFO_ROOM_m | FIFO_BYTE_m | DEV_CLR_m
	out	(phi_int_mask), a
	ret

enable_proc_abrt_int:
        ld	a, PROC_ABRT_m
	out	(phi_int_mask), a
	ld	a, INT_ENAB_m
	out	(phi_status), a
	ld	a, PROC_ABRT_m
	out	(phi_int_mask), a
	ret

; ******************** NMI PHI INT handler
do_phi_int:
        ex	af, af'
	exx
	in	a, (phi_status)
	push	af                              ; save the D0D1 bits?
	bit	LISTEN_ACT, (iy + YB(state_flags))
	jp	nz, phi_int_listen_active
	in	a, (phi_int_cond)
	push	af                              ; save the interrupting condition
	xor	a
	out	(phi_status), a
	ld	a, FIFO_ROOM_m | FIFO_BYTE_m | DEV_CLR_m
	out	(phi_int_mask), a
	ld	bc, 0
	ld	d, 38
.loop:	in	a, (phi_int_cond)
	bit	FIFO_ROOM, a
	jr	nz, out_fifo_room_available
	bit	DEV_CLR, a
	jr	nz, listen_dev_clr
	bit	FIFO_BYTE, a
	jr	nz, in_fifo_bytes_available
	in	a, (phi_status)
	bit	TLK_IDF, a
	jr	z, phi_int_listen_terminate
	djnz	.loop
	dec	c
	jr	nz, .loop
	dec	d
	jr	nz, .loop

phi_int_abort_listen:
        ld	hl, abort_listen                ; timed out waiting...
	jr	x1419

; not just FIFO_ROOM path?
out_fifo_room_available:
        pop	af                      ; restore the interrupt condition reg?
	out	(phi_int_cond), a
	call	enable_proc_abrt_int
	pop	af
	ex	af, af'
	ex	(sp), hl
	jr	nz, x1402               ; pre-interrupt flags??
	dec	hl                      ; dec interrupt return location by two
	dec	hl
x1402:	exx
	dec	hl                      ; retry the previous byte for the transfer
	inc	b
	exx
	ex	af, af'
do_phi_int_coda:
        ex	(sp), hl                ; set up return to (HL)
	out	(phi_status), a         ; restore the interrupt status
	ex	af, af'
	exx
	retn

listen_dev_clr:
        ld	hl, talkdone3
	jr	x1419

in_fifo_bytes_available:
        call	do_PPD
phi_int_listen_terminate:
        ld	hl, talkdone2
x1419:	call	set_parity_check_state
	pop	af
	out	(phi_int_cond), a
	pop	af
	jr	do_phi_int_coda

phi_int_listen_active:
        di
	in	a, (phi_int_cond)
	push	af
	xor	a
	out	(phi_status), a
	ld	a, FIFO_BYTE_m | DEV_CLR_m
	out	(phi_int_mask), a
	ld	bc, 0
	ld	d, 38
.loop:	in	a, (phi_int_cond)
	bit	FIFO_BYTE, a
	jr	nz, .in_fifo_bytes
	bit	DEV_CLR, a
	jr	nz, .dev_clr
	in	a, (phi_status)
	bit	FIFO_IDLE, a
	jr	z, phi_int_abort_listen
	djnz	.loop
	dec	c
	jr	nz, .loop
	dec	d
	jr	nz, .loop
	jr	phi_int_abort_listen

.in_fifo_bytes:
        ei
	jr	out_fifo_room_available

.dev_clr:
        ld	hl, x138a
	jp	x1419

;**************** END NMI PHI INT handler

receive_sector_data:
        xor	a                               ; A = 0 (HP_BYTES_PER_SECTOR, max do_listen_transfer)
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	z, .skip                        ; HP format
	ld	a, IBM_BYTES_PER_SECTOR
.skip:	ld	hl, data_buffer_count           ; large I/O buffer?
	jp	do_listen_transfer

cmd_terminate:
        call	terminate_talk
listen_terminate:
        call	prep_for_next_cmd
	call	deselect_drives
	call	do_PPE
	jp	wait_for_cmd_jv

abort_IO_program_error:
        call	check_DSJ_S1
	jr	nz, cmd_terminate
	ld	a, 0ah                          ; S1 = I/O program error
	jr	abort_common

abort_illegal_opcode:
        ld	a, 1                            ; S1 = illegal opcode
	jr	abort_common

abort_drive_fault_seek:
        set	4, (ix + DriveData.stat2_ds)    ; E (drive fault)
	set	7, (ix + DriveData.stat2_ds)    ; A (attention seek completed)
	jr	abort_drive_attention

abort_drive_fault:
        set	4, (ix + DriveData.stat2_ds)    ; E (drive fault)
	jr	abort_stat2_error

abort_seek_check:
        set	2, (ix + DriveData.stat2_ds)    ; C (seek check)
	set	7, (ix + DriveData.stat2_ds)    ; A (attention seek completed)
;x1498:
abort_drive_attention:
        ld	a, S1_DA                        ; drive attention
	jr	abort_common

abort_stat2_error:
        ld	a, S1_S2E                       ; Stat 2 error
abort_common:
        call	set_abort_S1
	jr	cmd_terminate


        ; set DSJ abort if not already DSJ_holdoff or DSJ_parity_error
        ; on entry A = S1 value
set_abort_S1:
        call	set_S1_and_unit
	ld	a, (DSJ)
	cp	DSJ_holdoff
	ret	p               ; return if DSJ_holdoff or DSJ_parity_error
	ld	(iy + YB(DSJ)), DSJ_aborted
	ret

normal_and_deselect_drives:
        call	deselect_drives ; A == 0 at return from deselect_drives
	ld	(DSJ), a         ; DSJ_normal

        ; on entry A = S1 value
set_S1_and_unit:
        ld	(S1), a
	ld	a, (current_unit)
	ld	(S1_unit), a
	ret

check_DSJ_S1:
        ld	a, (DSJ)
	or	a               ; DSJ_normal?
	ret	z               ; yes normal completion
	cp	DSJ_parity_error
	ret	nz              ; DSJ_aborted or DSJ_holdoff
	ld	a, (S1)
	or	a
	ret	z               ; normal
	cp	S1_DA
	ret	nz              ; no, other
	ld	a, (ix + DriveData.stat2_ds)
	and	STAT2_E_m | STAT2_F_m
	ret	z               ; why not just ret??
	ret

do_show_in_use_leds:
	push	af
	call	return_if_in_self_test
	push	hl
	ld	hl, (in_use_leds_stack_ptr)     ; push on the in_use_leds_stack
	inc	hl                              ; seems like this wastes one byte
	ld	(hl), a
set_leds:
        ld	(in_use_leds_stack_ptr), hl
	cpl                                     ; adjust for OC LED drivers
	out	(leds), a
	pop	hl
	pop	af
	ret

show_previous_in_use_leds:
        push	af
	call	return_if_in_self_test
	push	hl
	ld	hl, (in_use_leds_stack_ptr)
	dec	hl
	ld	a, (hl)
	jr	set_leds

return_if_in_self_test:
        bit	SELF_TEST_ACTIVE, (iy + YB(state_flags))
	ret	z                       ; no
	inc	sp                      ; pop our return address
	inc	sp
	pop	af                      ; pop the saved af value
	ret                             ; return to the caller's return address instead


;*************
; get_drive_stat2_bytes - fetch stat2 and stat2_ds
;
; on entry:
;       IX: DriveData pointer
;
; on exit:
;       D = stat2_ds
;       E = stat2
get_drive_stat2_bytes:
        ld	d, (ix + DriveData.stat2_ds)
	ld	e, (ix + DriveData.stat2)
	ret

; get pointer to per-drive data: 10 bytes each
; B = drive #
; IX = ptr (on return)
;x1507:
get_drive_data_ptr_in_ix:
        push	bc
	ld	ix, drive_data_table
	ld	c, 10            ; form index 10 * B
	xor	a
	cp	b
	jr	z, .done
.loop:	add	a, c
	djnz	.loop
	ld	c, a
	add	ix, bc           ; ix = &drive_data[10*B]
.done:	pop	bc
	ret

; prep_drive_regs - setup for working with a drive/disc
;
; on exit:
;       IX: DriveData pointer
;       DE: unchanged
;       (disc_format): the current disc format information
;       DriveData.xv, xv: set appropriately

prep_drive_regs:
        push	de
	ld	(current_unit), a       ; save unit #
	ld	(commanded_unit), a     ; save unit # again??
	ld	b, a
	call	get_drive_data_ptr_in_ix
	call	get_drive_stat2_bytes
	ld	b, FORMAT_HP
	bit	DT_HP, e
	jr	nz, .saveb
	ld	b, FORMAT_IBM
	bit	DT_IBM, e
	jr	nz, .saveb
	ld	b, FORMAT_UNKNOWN
.saveb:	ld	(iy + YB(disc_format)), b
	ld	a, (ix + DriveData.xv)
	and	DRV_SEL_m                       ; mask off PRECMP and HIDEN
	bit	FORMAT_IBM, b
	jr	nz, .skiphd                     ; IBM format only low density
	set	HIDEN, a
.skiphd:
        ld	(ix + DriveData.xv), a
	call	set_drv_xv
	pop	de
	ret

        ; returns with A==0
deselect_drives:
        in	a, (drv_status)
	bit	DISCHNG, a
	jr	z, deselect_drives_no_chng_check
	set	XV_SHDW_DISCHNG, (ix + DriveData.xv)
deselect_drives_no_chng_check:
        ld	a, NUNITS                               ; indicate no drive is selected
	ld	(commanded_unit), a
	xor	a
	out	(xv), a                                 ; deselect all drives
	ret

; head_unload_timer_check - see if it is time to unload heads on drives
;
; Keeps a head loaded for approximately 2 seconds if no further operations
; have been received. The time a head remains loaded may go as long as 10
; seconds on inactive drives if another drive has activity.
;
; on entry:
;       D = passed in parameter, approximate elapsed ticks

head_unload_timer_check:
        ld	a, (head_unload_set)
	and	DRV_SEL_m                       ; any active timers?
	ret	z                               ; no, nothing to do
	ld	c, a
	ld	b, NUNITS                       ; init unit number counter
	push	ix                              ; save the current drive_data_ptr
.loop:	srl	c
	dec	b
	jp	m, .done                        ; done with the loop
	jr	nc, .loop                       ; this drive's timer is not active, continue loop
	ld	a, (commanded_unit)
	cp	b
	jr	z, .loop                        ; skip the commanded drive
	call	get_drive_data_ptr_in_ix
	ld	a, (ix + DriveData.unload_ticks)
	sub	d
	ld	(ix + DriveData.unload_ticks), a
	jr	nc, .loop                       ; continue loop if ticks didn't underflow
	dec	(ix + DriveData.unload_timer)   ; approximately 1 sec has elapsed
	jp	p, .loop                        ; timer not exipred, continue loop
	call	maybe_unload_head
	ld	a, (commanded_unit)
	cp	NUNITS
	jr	z, .skip                        ; no drive is commanded, we're done
	pop	ix                              ; restore commanded drive's register values
	jp	set_drv_xv_disc_chng
.skip:	call	deselect_drives
.done:	pop	ix
	ret


;*************
; load_head_ix - Load the nead in the selected drive
;
; on entry:
;       IX: DriveData pointer
;
; on exit:
;       A, B: clobbered

do_load_head:
	ld	a, IU_LOAD
	set_in_use_leds
	call	load_head_ix
	ld	b, 60
	call	z, delay                        ; delay 60 ms if previously unloaded
	in	a, (drv_status)
	bit	READY, a
	jp	nz, show_previous_in_use_leds   ; the drive is ready (disc loaded)
	call	force_unload_head               ; drive not ready within 60 ms
	jp	abort_drive_fault

;*************
; load_head_ix - Load the head on the selected drive
;
; on entry:
;       IX: DriveData pointer
;
; on exit:
;       Z = drive was not in use (head unloaded)

load_head_ix:
        bit	HEADLOAD, (ix + DriveData.drv)  ; remember the previous state
	push	af
	set	HEADLOAD, (ix + DriveData.drv)  ; turn on IN-USE
	call	set_drv_xv_disc_chng
	bit	DOOR_LOCK, (ix + DriveData.drv) ; drive's door lock set?
	jr	nz, .skip
	ld	a, (head_unload_set)            ; add this drive to head_unload_set
	or	(ix + DriveData.xv)
	ld	(head_unload_set), a
	ld	(ix + DriveData.unload_timer), 24 ; initialize the unload timer (~24 sec?)
.skip:	pop	af                              ; return the previous IN-USE state
	ret


;*************
; maybe_unload_head - See if we are allowed to unload the head and if so, do it.
; on entry:
;       B = drive unit number
;
; on exit:
;       IX: DriveData pointer

maybe_unload_head:
        call	get_drive_data_ptr_in_ix
	bit	DOOR_LOCK, (ix + DriveData.drv)
	jr	nz, remove_from_head_unload_set

;*************
; force_unload_head - Unload the head.
; on entry:
;       IX: DriveData pointer
;
; on exit:
;       IX: DriveData pointer

force_unload_head:
        res	HEADLOAD, (ix + DriveData.drv)
	call	set_drv_xv_disc_chng            ; unload the head

remove_from_head_unload_set:
        ld	a, (ix + DriveData.xv)
	cpl                                     ; form mask
	and	(iy + YB(head_unload_set))      ; remove this drive from head_unload_set
	ld	(head_unload_set), a
	ret


;*************
; Notes:
;       If HEADLOAD is active when this is called it will lock the door.
;       If HEADLOAD is inactive when this is called it will unlock the door.
x15f3:
set_drv_xv_disc_chng:
        in	a, (drv_status)
	bit	DISCHNG, a
	jr	z, set_drv_xv                           ; no change has been detected
	set	XV_SHDW_DISCHNG, (ix + DriveData.xv)    ; remember disc change was detected
set_drv_xv:
        xor	a
	out	(xv), a                         ; deselect all drives
	ld	a, (ix + DriveData.drv)         ; set this drive's drv value
	out	(drv), a                        ; HEADLOAD (aka IN-USE) state is latched here
	ld	a, (ix + DriveData.xv)          ; select the drive
	out	(xv), a

do_deassert_in_use:
	ld	a, (ix + DriveData.drv)
	and	~HEADLOAD                       ; Turn off the drive's In Use signal
	out	(drv), a
	ret

;*************
; get_drive_SS - Determine the stat2_ds SS value from the currently selected
;          drive's status register or a remembered DISCHNG status.
;
; on entry:
;       IX: DriveData pointer
;
; on exit:
;       IX: unchanged
;       Z: A is valid (SS value)
;       C: drive ready + first status

;x1612:
get_drive_SS:
        in	a, (drv_status)
	bit	DISCHNG, a
	jr	nz, .dchng
	bit	XV_SHDW_DISCHNG, (ix + DriveData.xv)
	jr	nz, .dchng
	bit	READY, a
	jr	nz, .check_ready

.no_disc:
        ld	a, SS_NO_DISC
	res	DOOR_LOCK, (ix + DriveData.drv)
	cp	a
	ret                             ; NC, Z, A = SS_NO_DISC

.dchng:	res	XV_SHDW_DISCHNG, (ix + DriveData.xv)
	bit	READY, a
	jr	z, .no_disc

.ready:	xor	a                       ; A = 0
	scf
	ret                             ; C, Z, A = SS_READY

.check_ready:
        bit	SS_NOT_READY_m, (ix + DriveData.stat2_ds)
	jr	nz, .ready
	or	a                       ; A is non-zero here
	scf
	ret                             ; C, NZ, A = drv_status

;*************
; validate_sector_number - check if the sector number is in range
;
; on entry:
;       B: sector number
;
; on exit:
;       A: clobbered
;       Z: valid
;       NZ: invalid

validate_sector_number:
        push	hl
	ld	hl, sector_range_tbl
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	z, .skip
	inc	hl                      ; choose the IBM values
.skip:	ld	a, b
	cp	(hl)
	jr	c, .invalid             ; borrow, was < min number
	inc	hl                      ; now check the max number
	inc	hl
	cp	(hl)
	jr	z, .invalid
	jr	nc, .invalid            ; no borrow, was > max number
	xor	a                       ; set Z
.invalid:
        pop	hl
	ret

sector_range_tbl:
        db      HP_FIRST_SECTOR,        IBM_FIRST_SECTOR
        db      HP_SECTORS_PER_TRACK,   IBM_SECTORS_PER_TRACK

set_lowcurr_and_precmp:
        ld	a, (ix + DriveData.physical_cylinder)
	bit	FORMAT_IBM, (iy + YB(disc_format))
	jr	nz, .skipp
	res	PRECMP, (ix + DriveData.xv)
	cp	HP_PRECMP_START_TRACK
	jr	c, .skipp
	set	PRECMP, (ix + DriveData.xv)
.skipp:
        res	LOWCURR, (ix + DriveData.drv)
	cp	HP_LOWCURR_START_TRACK
	jr	c, .skipl
	set	LOWCURR, (ix + DriveData.drv)
.skipl:	deassert_in_use
	ld	a, (ix + DriveData.xv)
	out	(xv), a
	ret

;*************
; get_doubled_sided_bit - get the drive's TWO-SIDED signal
;
; on entry:
;       E = copy of DriveData.stat2
;
; on exit:
;       E = copy of DriveData.stat2 with DT_DS set appropriately

get_double_sided_bit:
        res	DT_DS, e                ; assume single sided
	in	a, (drv_status)
	bit	TWOSIDE, a
	ret	z
	set	DT_DS, e
	ret

;x168d:
; media_type_check - determine the disc media type
;
; on entry:
;       DE: (zeroed?)
;
; on exit:
;       D: DriveData.stat2_ds
;       E: DriveData.stat2
media_type_check:
        push	de
	ld	a, DT_HP_m                      ; assume HP format
	call	set_stat2_check_ready
	jr	z, .notready
	reset_drive
	jr	z, .ready
	pop	de
	set	STAT2_C, d                      ; set stat2 C bit (seek check)?
	jr	.inc_a_ret

.ready:	ld	(iy + YB(600eh)), 5             ; retries?
.retry:	ld	a, DT_HP_m
	call	set_stat2_check_ready
	jr	z, .notready
	wait_for_sector
	ld	a, DT_HP_m
	jr	z, .exit
	pop	de                              ; get stat2 and stat2_ds back
	push	de
	bit	DT_DS, e
	jr	nz, .twoside
	ld	a, DT_IBM_m
	call	set_stat2_check_ready
	jr	z, .notready
	wait_for_sector
	jr	nz, .twoside
	ld	(ix + DriveData.sector), 1      ; IBM format first sector is 1
	ld	a, DT_IBM_m
	jr	.exit

.twoside:
        dec	(iy + YB(600eh))
	ld	a, DT_BLANK_m
	jr	z, .exit                        ; retries expired?
	bit	DT_DS, e
	jr	z, .oneside
	bit	HEAD1, (ix + DriveData.drv)     ; switch to the other head
	set	HEAD1, (ix + DriveData.drv)
	jr	z, .skip                        ; HEAD1 was not previously set
	res	HEAD1, (ix + DriveData.drv)
.skip:	deassert_in_use
	bit	HEAD1, (ix + DriveData.drv)
	jr	nz, .retry                      ; now try again with HEAD1

.oneside:
        ld	c, 0                            ; step in towards center
	call	step_drive_jv
	inc	(ix + DriveData.physical_cylinder)
	jr	.retry

.exit:	pop	de
	push	af
	or	e                               ; set the disc type in stat2 (in E)
	ld	e, a
	pop	af                              ; return with the disc type in A
	ret

.notready:
        pop	de
	set	STAT2_E, d
.inc_a_ret:
        inc	a                               ; set STAT2_RSVD ??
	ret

;******************************************************************
; set_stat2_check_ready - set the stat2 value and check if the
;                         drive is ready
; on entry:
;       A: the stat2 value to set
;
; on exit:
;      NZ: the drive is ready
;       Z: not ready

set_stat2_check_ready:
        ld	(ix + DriveData.stat2), a
	call	deselect_drives
	ld	a, (current_unit)
	call	prep_drive_regs
	call	load_head_ix
	in	a, (drv_status)
	bit	READY, a
	ret	nz                      ; why not just ret?
	ret	z


	fillto	1720h, 076h

do_reset:
	jr	cold_start

do_initiate_self_test:
        jr	reset_peripherals

cold_start:
        xor	a
	ld	d, a
	out	(leds), a               ; turn on all LEDs
	set	ERROR, a
	out	(monitor_U6), a         ; turn on monitor ERROR LED
	in	a, (switches)
	and	WRTEST_m
	ld	e, a
	set	SELFTEST, e             ; DE = bit 4: SELF TEST (internal, not commanded), bit 3: WR TEST
 
reset_peripherals:
        di
	ld	a, PROGRES
	out	(reset), a 

        ; run off the end of test_jmptbl?? or initiate self test
test_1738: 
        xor	a
	out	(cntl), a        ; clear cntl
	ld	sp, ramend
	jp	cpu_test

; warm start??
x1741:	ld	iy, ramstart             ; IY convenient pointer to start of RAM
	bit	6, (iy + YB(test_jmptbl_ptr))   ; (x6015) bit 6 set? (after ram test this should be zero)
	jr	nz, do_self_test_loop    ; Yes, run self tests
	ld	b, 0
x174d:	call	get_drive_data_ptr_in_ix
	push	bc
	inc	b
	ld	a, 10h
x1754:	rrca
	djnz	x1754
	ld	(ix + DriveData.xv), a
	push	af
	out	(xv), a
	reset_drive
	jr	z, x1769
	pop	af
	or	(iy + YB(cmd_count))
	ld	(cmd_count), a
	jr	x1777

x1769:	ld	c, 0             ; step in towards center
	call	step_drive_jv
	call	delay_20ms
	ld	b, 4
	call	x1c15
	pop	af
x1777:	pop	bc
	inc	b
	bit	2, b
	jr	z, x174d
	xor	a
	out	(xv), a
	set	6, (iy + YB(test_flags))

do_self_test_loop:
        ld	bc, test_jmptbl          ; Initialize the self test table pointer
	ld	(test_jmptbl_ptr), bc
	ld	hl, state_flags
	set	SELF_TEST_ACTIVE, (hl)
self_test_loop:
        ld	iy, (test_jmptbl_ptr)    ; Get the address of the next test
	ld	l, (iy + YB(ramstart))
	inc	iy
	ld	h, (iy + YB(ramstart))
	inc	iy
	ld	(test_jmptbl_ptr), iy    ; save new pointer value
	ld	iy, ramstart             ; restore IY to usual value
	ld	bc, x17ab                ; Set test return address
	push	bc
	jp	(hl)                    ; Run the next self test

x17ab:	jr	self_test_loop

delay_20ms:
        ld	b, 14h
	jp	delay

test_jmptbl:
        dw timeout_bit_test
        dw overrun_bit_test
        dw ibm_data_loop_test
        dw hp_data_loop_test
        dw crc_test
seek_recalibrate_test_ptr:
        dw seek_recalibrate_test
        dw rotational_timing_test
        dw write_test
        dw write_test_part2
        dw write_test_part3
read_test_ptr:
        dw read_test
test_1e9c_ptr:
        dw test_1e9c
test_1ef3_ptr:
        dw test_1ef3
test_1ed9_ptr:
        dw test_1ed9
        dw test_1738

        ; update_monitor_leds data 1 = on
        ; A = monitor_U6 value
        ; C = leds value (mirrored by the montor board)
update_monitor_leds:
        ld	b, a
	ld	(leds_shdw), bc         ; Save them in the shadows
	cpl                             ; Complement for OC drivers
	out	(monitor_U6), a
	ld	a, c
	cpl                             ; Complement for OC drivers
	out	(leds), a
x17dc:	ret


led_data_to_a_c:
        ld	bc, (leds_shdw)
	ld	a, b
	ret

	;on entry from cold_start: DE = bit 4: SELF TEST (artificially set), bit 3: WR TEST
cpu_test:
        ld	a, 0c0h                 ; Processor test 15 ST 1
	out	(leds), a
	ld	a, 0ffh
	out	(monitor_U6), a
	ld	a, 0
	add	a, a
	jr	z, test_nz              ; 0 better be zero!
hang_z_not_set:
        jr	hang_z_not_set

test_nz:
hang_nz_set:
        jr	nz, hang_nz_set

	add	a, 0fh                  ; A = 15
	jp	p, test_m               ; 15 better be positive!
hang_p_not_set:
        jr	hang_p_not_set

test_m:
hang_m_set:
        jp	m, hang_m_set
	cp	0eh
hang_cp_pos:
        jp	m, hang_cp_pos          ; 15 - 14 better be positive!

	cp	0fh
hang_cp_ne:
        jr	nz, hang_cp_ne          ; 15 better equal 15!

x1807:	cp	10h
hang_cp_neg:
        jp	p, hang_cp_neg          ; 15 - 16 better be negative!

	sub	10h                     ; A = -1
	jp	m, hang_sub_pos         ; subtract better work the same way
hang_sub_neg:
        jr	hang_sub_neg

hang_sub_pos:
        jp	p, hang_sub_pos         ; 15 - 16 should not also be positive!

        ; Do some ALU and register transfer tests
	ld	b, a             ; B = -1
	ld	c, b             ; C = -1
	ld	h, c             ; H = -1
	ld	l, h             ; L = -1
	inc	l                ; L = 0
	ld	h, l             ; H = 0
	ld	c, h             ; C = 0
	ld	b, c             ; B = 0
	ld	a, b             ; A = 0
	dec	a                ; A = 0ffh
	and	5ah              ; A = 5ah
	ld	b, a             ; B = 5ah
	or	99h              ; A = 0dbh
	ld	c, a             ; C = 0dbh
	xor	0dbh             ; A = 0
	cpl                      ; A = 0ffh
	res	3, a             ; A = 0f7h
	ld	h, a             ; H = 0f7h
	scf                      ; CF=1
	adc	a, b             ; A = 52h, CF=1
	ld	l, a             ; L = 52h
	ccf                      ; CF=0
	sbc	a, c             ; A = 77h, CF=1
	set	7, a             ; A = 0f7h
	neg                      ; A = 09h, CF=1
	rrca                     ; A = 04h, CF=1
	rl	b                ; B = 0b5h, CF=0
	rlc	c                ; C = 0b7h, CF=1
	rr	h                ; H = 7bh, CF=1
	sla	l                ; L = 0a4h, CF=0
	sra	a                ; A = 02h, CF=0
	srl	h                ; H = 3dh, CF=1
	jr	c, hang_alu_nc   ; all that mess should end up with the C flag set
hang_alu_c:
        jr	hang_alu_c

hang_alu_nc:
        jr	nc, hang_alu_nc
	add	a, b
	add	a, c
	add	a, h
	add	a, l
	ld	b, d                    ; save DE contents into BC
	ld	c, e
	ld	d, 54h
	add	a, d
	ld	e, 0a2h
	set	0, e
	cp	e
hang_cpu_nz:
        jr	nz, hang_cpu_nz

	bit	6, h                    ; H = 3dh from above
hang_bit_test:
        jr	z, hang_bit_test
	ld	d, b                    ; restore DE
	ld	e, c
	ld	hl, processor_st2       ; test indirect jump
	jp	(hl)

hang_ind_jp:
        jr	hang_ind_jp


	; DE = bit 4: SELF TEST (artificially set), bit 3: WR TEST
processor_st2:
	ld	a, 0a0h                 ; Processor test ST 2
	out	(leds), a
	ld	a, 0ffh
	out	(monitor_U6), a
	ld	b, 0a5h
	push	bc
	pop	af
	cp	b
hang_push_pop:
        jr	nz, hang_push_pop        ; A == B?

	xor	a
	ld	hl, highest_stack_word   ; Reg B containing a5h pushed on stack previously
	ld	a, (hl)
	cp	b
hang_stack_rw:
        jr	nz, hang_stack_rw        ; A == B?

	ld	ix, highest_stack_word   ; Test index register accesses
	ld	iy, penultimate_stack_word
	cp	(ix + DriveData.physical_cylinder)
hang_ix:
        jr	nz, hang_ix

	cp	(iy + YB(S1_unit))
hang_iy:
        jr	nz, hang_iy

	call	call_ret_test
hang_call_ret:
        jr	hang_call_ret

	jp	romcsum                 ; CPU test complete

call_ret_test:
        ld	bc, hang_call_ret
	ld	a, (hl)
	cp	b
hang_ret_addr_H:
        jr	nz, hang_ret_addr_H
	dec	hl
	ld	a, (hl)
	cp	c
hang_ret_addr_L:
        jr	nz, hang_ret_addr_L
	inc	(hl)                    ; adjust ret addr to return to jp romcsum
	inc	(hl)
	ret

	; DE = bit 4: SELF TEST (artificially set), bit 3: WR TEST
romcsum:
        ld	a, 0fch                 ; ROM test ST 0
	out	(leds), a
	ld	a, 0ffh
	out	(monitor_U6), a

	ld	ix, rom_csum_L-1        ; highest ROM address to check
	ld	bc, rom_csum_L          ; # ROM bytes to check (excl ROM csum bytes)
	push	de                      ; save DE (written to test_flags later)
	xor	a
	ld	d, a
	ld	h, a
	ld	l, a

romcsum_loop:
	dec	bc
	ld	e, (ix + DriveData.physical_cylinder)
	add	hl, de
	dec	ix
	cp	b
	jr	nz, romcsum_loop
	cp	c
	jr	nz, romcsum_loop
	ld	a, (rom_csum_L)
	cp	l
romcsum_fail_l:
	jr	nz, romcsum_fail_l

	ld	a, (rom_csum_H)
	cp	h
romcsum_fail_h:
	jr	nz, romcsum_fail_h

	pop	de                      ; restore DE (written to test_flags later)

ramtest:
	ld	hl, ramstart            ; Fill RAM with an address dependent pattern
	ld	bc, ramsize
fill_loop:
        ld	a, l
	add	a, h
	ld	(hl), a
	inc	hl
	dec	c
	jr	nz, fill_loop
	djnz	fill_loop

	ld	hl, ramstart             ; Check RAM for the address dependent pattern
	ld	bc, ramsize
chk_loop:
        ld	a, l
	add	a, h
	cp	(hl)
	jr	nz, RAM_addr_fail
	inc	hl
	dec	c
	jr	nz, chk_loop
	djnz	chk_loop
	ld	iy, ram_test_patterns    ; Test with the static patterns

do_pat_tests:
        ld	a, (iy + YB(ramstart))
	call	do_pat_test
	inc	iy
	or	a
	jr	nz, do_pat_tests
	ld	(test_flags), de        ; DE = bit 4: SELF TEST (artificially set cold start), bit 3: WR TEST
	jr	do_phi_test             ; ram test done


        ; Test pattern in A also at (iy + YB(ramstart))
do_pat_test:
        pop	ix                      ; Save return addr since we will test that loc
	ld	bc, ramsize
	ld	hl, ramstart
pfill_loop:
        ld	(hl), a
	inc	hl
	dec	c
	jr	nz, pfill_loop
	djnz	pfill_loop

	ld	bc, ramsize
	ld	hl, ramstart
pchk_loop:
        ld	a, (hl)
	cp	(iy + YB(ramstart))
	jr	nz, pattern_test_fail
	inc	hl
	dec	c
	jr	nz, pchk_loop
	djnz	pchk_loop
	push	ix                      ; Restore the return addr
	ret

        ; A = failed RAM read data
pattern_test_fail:
        and	0f0h
	ld	c, a
	ld	a, (iy + YB(ramstart))  ; Get the correct data
	and	0f0h
	cp	c
	ld	a, 0f4h                 ; Upper nibble failed U36 test 5 + DONE
	jr	nz, x193a
	ld	a, 0f2h                 ; Lower nibble failed U35 test 6 + DONE
x193a:	out	(leds), a               ; NOTE: LED pattern 0 = on here
	ld	a, 0ffh
	out	(monitor_U6), a
hang_RAM_failure:
        jr	hang_RAM_failure

RAM_addr_fail:
        ld	a, 0f6h                  ; RAM address fail test 4 + DONE
	out	(leds), a                ; NOTE: LED pattern 0 = on here
	ld	a, 0ffh
	out	(monitor_U6), a
hang_RAM_addr_fail:
        jr	hang_RAM_addr_fail

ram_test_patterns:
        db 0ffh
        db 0a5h
        db 05ah
        db 000h                          ; Last RAM test, leave it cleared

do_phi_test:
        ld	a, 0                     ; PHI Test subtest 0
	ld	c, 0fh
	call	update_monitor_leds
	xor	a
	out	(phi_status), a          ; D0D1 = 00
	ld	a, EIGHT_BIT_m
	out	(phi_control), a
	out	(phi_control), a	; why twice?
                                        ; I'm guessing so that D0D1 get pulled from phi_status,
                                        ; the first time they may be floating or random?
                                        ; Since D0D1 do not have documented meanings in phi_control,
                                        ; perhaps they found an issue with testing that requires it.

        ; set up Amigo identify response bytes 00h 81h (9895A)
	ld	a, 0
	out	(phi_id1), a
	ld	a, 81h
	out	(phi_id2), a

	xor	a
	out	(phi_status), a          ; D0D1 = 00
	ld	a, 7fh
	out	(phi_int_mask), a        ; disable interrupts, enable most interrupt conditions
	xor	a
	out	(phi_status), a          ; D0D1 = 00
	ld	a, 20h
	out	(phi_address), a         ; listen always, offline (diag mode)
	ld	a, 41h
	out	(phi_int_cond), a        ; reset device clear and processor handshake abort bits
.loop:	in	a, (phi_int_cond)        ; clear out the inbound FIFO
	bit	FIFO_BYTE, a
	jr	z, .empty
	in	a, (phi_fifo)
	jr	.loop

        ; Note: in the offline diag mode the PHI chip behaves as it is a self-contained
        ; HP-IB bus with a system controller and a device connected internal to the chip.
        ; When in this diag mode the chip assumes system controller status regardless
        ; of the state of the SCTRL pin. In diag mode the chip responds to address 30 (1eh).

.empty:	ld	a, DATA_FRZ_m            ; D0D1=00, reset outbound fifo freeze bit
	out	(phi_status), a
	ld	a, 10h
	out	(phi_control), a         ; assert IFC (disable 8-bit processor mode??)
	ld	a, 89h
	out	(phi_control), a         ; 8-bit processor, respond to PPOL, init fifo
	ld	d, 0ah
	ld	c, 0
	call	rd_phi_int_cond_cp_cd   ; FIFO room available and idle?
	ld	a, 40h                   ; D0D1=01 (send interface commands)
	out	(phi_status), a
	ld	a, 5fh                   ; send UNT
	out	(phi_fifo), a
	ld	a, 7eh                   ; send SAD MLA
	out	(phi_fifo), a
	ld	a, 0c0h                  ; D0D1=11
	out	(phi_status), a
	ld	a, 2                     ; set "uncounted transfer" 2 bytes - read our own id bytes
	out	(phi_fifo), a
	ld	b, 1
	call	delay                   ; delay 1 ms
	ld	d, 0eh
	call	rd_phi_int_cond_cp_cd   ; FIFO room available, bytes available and idle?
	ld	c, 0
	ld	d, 0
	call	rd_phi_fifo_cp_cd       ; first id byte matches?
	ld	c, 0c0h
	ld	d, 81h
	call	rd_phi_fifo_cp_cd       ; second id byte matches?
	ld	c, 0
	ld	d, 0ah
	call	rd_phi_int_cond_cp_cd   ; now FIFO room avilable and idle? (we read the fifo)
	xor	a                       ; D0D1=00
	out	(phi_status), a
	out	(phi_address), a         ; reset listen always, offline
	ld	a, 40h
	out	(phi_status), a          ; D0D1=01 (send interface commands)
	ld	a, 3fh                   ; send UNL
	out	(phi_fifo), a
	ld	a, 49h                   ; send TALK 9?
	out	(phi_fifo), a
	ld	a, 9                     ; send TCT (take control)
	out	(phi_fifo), a
	call	rd_phi_int_cond_cp_cd   ; FIFO room available, device clear?
	xor	a                       ; D0D1=00
	out	(phi_status), a
	ld	a, 40h                   ; Talk always
	out	(phi_address), a
	ld	a, 60h                   ; Talks always, listen always
	out	(phi_address), a
	ld	a, 0ffh                  ; send 16 bytes to the outbound FIFO (fill both FIFOs)
	ld	b, 10h                   ; verifies handshaking
x19f1:	out	(phi_fifo), a
	dec	a
	djnz	x19f1
	ld	c, 0
	ld	d, 4
	call	rd_phi_int_cond_cp_cd   ; FIFO bytes available?

	ld	c, 0                     ; Read the 16 bytes from the inbound FIFO
	ld	d, 0ffh
	ld	b, 10h
x1a03:	call	rd_phi_fifo_cp_cd       ; Does each byte match?
	dec	d
	djnz	x1a03

	ld	c, 0
	ld	d, 0ah
	call	rd_phi_int_cond_cp_cd   ; FIFO room available, idle?

	xor	a                       ; D0D1=00
	out	(phi_status), a
	ld	a, 81h			; set 8-bit processor and initialize outbound FIFO
	out	(phi_control), a

	jp	x1741


rd_phi_int_cond_cp_cd:
        in	a, (phi_int_cond)
	jr	x1a20

rd_phi_fifo_cp_cd:
        in	a, (phi_fifo)
x1a20:	cp	d
hang_phi_reg_miscompare:
        jr	nz, hang_phi_reg_miscompare

	in	a, (phi_status)	; check high order bits
	and	0c0h
	cp	c
x1a28:	jr	nz, x1a28
	ret

timeout_bit_test:
	ld	a, 0                    ; reset subtest
	ld	c, 11h                  ; Time-out bit test
	call	update_monitor_leds
	ld	a, (test_flags)
	and	3                       ; Low two bits indicate what?
	jr	nz, x1a3d
	set	1, (iy + YB(test_flags))
x1a3d:	ld	b, 250
	ld	a, r_TIMEOUT_m
	out	(reset), a              ; reset TIMEOUT
	call	delay                   ; delay 250 ms
	out	(reset), a              ; reset TIMEOUT
	call	delay                   ; delay 256 ms
	out	(reset), a              ; reset TIMEOUT
	call	delay                   ; delay 256 ms
	ld	b, 100
	call	delay                   ; delay 100 ms
	in	a, (switches)
	bit	TIMEOUT, a
	jp	nz, tsterr_st0          ; timeout high when it should be low ST 0
	call	delay                   ; delay 256 ms (B=0 here)
	ld	b, 250
	call	delay                   ; delay 250 ms
	in	a, (switches)
	bit	TIMEOUT, a
	jp	z, tsterr_st1           ; timeout low when it should be high ST 1
	ret

overrun_bit_test:
	ld	a, 0                    ; reset subtest
	ld	c, 11h                  ; Overrun bit test
	call	update_monitor_leds
	xor	a
	out	(xv), a                 ; deselect all drives
	ld	a, WRITON_m
	out	(cntl), a               ; WRITON but not WRITDRV
	out	(data), a               ; set a data byte (value unimportant)
	ld	a, r_OVERUN_m
	out	(reset), a              ; reset OVERUN
	in	a, (drv_status)
	bit	OVERUN, a
	jp	nz, tsterr_st2          ; overrun high when it should be low ST 2
	ld	b, 0ah
.loop:	djnz	.loop
	in	a, (drv_status)
	bit	OVERUN, a
	jp	z, tsterr_st3           ; overrun low when it should be high ST 3
	xor	a
	out	(cntl), a
	ret

ibm_data_loop_test:
	ld	a, TIMEOUT_m
	out	(reset), a
	ld	a, 0                     ; IBM Data Loop Test ST 1
	ld	c, 13h
	call	update_monitor_leds
	ld	a, WRITON_m | CRCON_m
	out	(cntl), a
	ld	a, 0e7h
	out	(clock), a
	ld	a, 0cah
	out	(data), a
	in	a, (data)
	cp	0cah
	jp	nz, tsterr_st0          ; data byte read error
	in	a, (clock)
	cp	0e7h
	jp	nz, tsterr_st1          ; clock byte read error
	xor	a
	out	(cntl), a
	ret

hp_data_loop_test:
	ld	a, TIMEOUT_m
	out	(reset), a
	ld	a, 0                    ; HP Data Loop Test
	ld	c, 13h
	call	update_monitor_leds
	ld	a, HIDEN_m
	out	(xv), a
	ld	l, 0                    ; normal clock for the AM byte?
	call	.dotest
	jr	nz, .failed
	ld	a, l
	cp	28h                     ; did the clock register have the right value?
	jp	nz, tsterr_st3          ; no, HP Data Loop Test ST 3 failed clock byte read
	ld	l, 38h                  ; poison the clock for the AM byte?
	call	.dotest
	jr	nz, .failed
	xor	a
	out	(cntl), a               ; turn off stuff
	ret

.failed:
        bit	3, a
	jp	nz, tsterr_st4          ; Address Mark Test ST 4 failed AM on when it should be off
	bit	2, a
	jp	nz, tsterr_st5          ; Address Mark Test ST 5 failed AM off when it should be on
	jp	tsterr_st2              ; HP Data Loop Test ST 2 failed data byte read?

; on entry:
;       L: clock register value to use with the AM byte
;
; on exit:
;       Z: success
;       L: clock register value when the AM was read
.dotest:
        xor	a                       ; A = 0
	out	(cntl), a
	out	(clock), a
	ld	a, WRITON_m | CRCON_m
	out	(cntl), a
	ld	a, 0cch
	out	(data), a
	ld	b, 2
	ld	c, data
	ld	d, 0ffh                 ; clock sync preamble?
	ld	e, HP_DATA_AM_DPAT
.loop:	out	(c), d
	djnz	.loop
	ld	a, READON_m | WRITON_m | CRCON_m
	out	(cntl), a
	out	(c), d
	out	(c), d
	in	a, (switches)
	and	AMDT_m
	jr	nz, .set_a_bit_3        ; st 4 failed
	ld	a, l
	out	(clock), a              ; write the callers clock value
	out	(c), e                  ; write address mark byte
	in	a, (switches)
	in	b, (c)                  ; read data register
	ld	h, a
	in	a, (clock)
	ld	l, a
	xor	a
	out	(clock), a
	out	(c), e
	bit	AMDT, h
	jr	z, .set_a_bit_2         ; st 5 failed
	ld	a, b
	cp	43h
	jr	nz, .set_a_bit_0        ; st 2 failed
	out	(c), e
	ld	a, WRITON_m | CRCON_m
	out	(cntl), a
	xor	a                       ; A = 0
.ret:	and	0ffh
	out	(c), e
	ret

.set_a_bit_3:
        ld	a, 8
	jr	.ret

.set_a_bit_2:
        ld	a, 4
	jr	.ret

.set_a_bit_0
        ld	a, 1
	jr	.ret

crc_test:
	ld	a, 1             ; reset TIMEOUT
	out	(reset), a
	ld	a, 0             ; CRC Test ST 0
	ld	c, 15h
	call	update_monitor_leds
	xor	a
	out	(xv), a
	ld	a, 4
	out	(cntl), a
	out	(data), a
	ld	a, 24h
	out	(cntl), a
	ld	a, 5ch
	out	(data), a
	in	a, (drv_status)
	bit	5, a
	jp	z, tsterr_st0
	ld	a, 56h
	out	(data), a
	ld	a, 36h
	out	(cntl), a
	in	a, (data)
	in	a, (data)
	cp	6ch
	jp	nz, tsterr_st1
	in	a, (data)
	cp	0eeh
	jp	nz, tsterr_st1
	in	a, (drv_status)
	bit	5, a
	jp	nz, tsterr_st2
	xor	a
	out	(cntl), a
	ld	a, (test_flags)
	dec	a
	ld	(test_flags), a
	and	3
	jr	nz, reset_test_jump_table_ptr
	ret

reset_test_jump_table_ptr:
        ld	hl, test_jmptbl
	ld	(test_jmptbl_ptr), hl
	ret

seek_recalibrate_test:
	ld	a, (current_unit)
	call	prep_drive_regs
	ld	a, (current_unit)
	rlca
	rlca
	rlca
	rlca
	ld	c, 17h
	call	update_monitor_leds
	ld	a, (cmd_count)
	and	(ix + DriveData.xv)
	jr	z, x1bc6
	ld	hl, test_1e9c_ptr
	ld	(test_jmptbl_ptr), hl
	ret

x1bc6:	set	5, (iy + YB(test_flags))
	bit	4, (iy + YB(test_flags))
	jr	z, x1bd5
	ld	a, (current_head)
	or	a
	ret	nz
x1bd5:	reset_drive
	jp	nz, tsterr_st3
	call	load_head_ix
	ld	c, 0
	ld	b, 4ch
	call	x1c15
	jp	nz, tsterr_st4
	call	delay_20ms
	dec	c
	ld	b, 4bh
	call	x1c15
	jp	nz, tsterr_st5
	ld	b, 1
	call	x1c15
	jp	z, tsterr_st6
	bit	4, (iy + YB(test_flags))
	ret	nz
	xor	a
	ld	c, a
	ld	b, (iy +14h)
	ld	(ix + DriveData.physical_cylinder), b
	cp	b
	ret	z
	push	bc
	call	delay_20ms
	pop	bc
	call	x1c15
	jp	nz, tsterr_st4
	ret	z

x1c15:	call	step_drive_jv
	in	a, (drv_status)
	bit	2, a                     ; reached track 0?
	ret	nz                      ; yes
	djnz	x1c15
	ret

rotational_timing_test:
	ld	a, (monitor_U6_shdw)
	ld	c, 19h
	call	update_monitor_leds
	in	a, (drv_status)
	bit	4, a
	ret	z
	xor	a
	ld	c, 62h
	call	x1c49
	jp	z, tsterr_st0
	call	x1c49
	ld	bc, 031e8h
	ld	de, 02e87h
	call	x1c64
	ret	z
	jp	m, tsterr_st1
	jp	p, tsterr_st2
x1c49:	ld	l, a
	ld	h, a
x1c4b:	inc	hl
	cp	l
	jr	nz, x1c51
	cp	h
	ret	z
x1c51:	in	b, (c)
	bit	0, b
	jr	nz, x1c4b
x1c57:	inc	hl
	cp	l
	jr	nz, x1c5d
	cp	h
	ret	z
x1c5d:	in	b, (c)
	bit	0, b
	jr	z, x1c57
	ret

x1c64:	push	hl
	scf
	ccf
	sbc	hl, bc
	pop	hl
	ret	p
	push	hl
	scf
	ccf
	sbc	hl, de
	pop	hl
	ret	m
	xor	a
	ret

write_test:
	ld	a, (monitor_U6_shdw)
	ld	c, 1bh
	call	update_monitor_leds
	bit	3, (iy + YB(test_flags))
	jr	nz, x1c89
	ld	hl, read_test_ptr
	ld	(test_jmptbl_ptr), hl
	ret

x1c89:	in	a, (drv_status)
	bit	4, a
	jp	z, tsterr_st1
	bit	3, a
	jp	nz, tsterr_st2
	bit	4, (iy + YB(test_flags))
	ret	z
	ld	a, (current_head)
	and	BYTENOT(HEAD1_m)
	ld	(ix + DriveData.physical_cylinder), a
	ret	z
	ld	c, 1
	call	step_drive_jv
	jp	delay_20ms

write_test_part2:
	ld	a, (monitor_U6_shdw)
	ld	c, 1bh
	call	update_monitor_leds
	call	sel_head_0
	call	x1d47
	call	fmt_ibm_track_jv
	jp	nz, tsterr_st0
	ld	a, (monitor_U6_shdw)
	ld	c, 1dh
	call	update_monitor_leds
	ld	c, 40h
	ld	b, 1
x1ccb:	call	x1d98
	call	x1cd4
	jr	nz, x1ccb
	ret

x1cd4:	inc	b
	ld	a, b
	cp	1bh
	ret

write_test_part3:
	ld	a, (monitor_U6_shdw)
	set	3, a
	ld	c, 1bh
	call	update_monitor_leds
	call	sel_head_0
	call	x1d37
	call	fmt_hp_track
	jp	nz, tsterr_st0
	ld	e, (ix + DriveData.stat2)
	ld	d, (ix + DriveData.stat2_ds)
	call	get_double_sided_bit
	jr	z, x1d12
	call	sel_head_1
	set	DT_DS, (ix + DriveData.stat2)
	call	led_data_to_a_c                 ; A = monitor_U6_shdw
	set	HEAD, a                         ; turn on head LED on monitor board
	call	update_monitor_leds
	call	x1d37
	call	fmt_hp_track
	jp	nz, tsterr_st0
x1d12:	ld	h, 0
	call	x1d78
x1d17:	call	x1d98
	call	x1d32
	jr	nz, x1d17
	ld	h, 40h
	call	x1d78
	ret	z
x1d25:	call	x1d98
	call	x1d32
	jr	nz, x1d25
	res	7, (iy + 14h)
	ret

x1d32:	inc	b
	ld	a, b
	cp	1eh
	ret

x1d37:	ld	a, (ix + DriveData.stat2)
	and	0edh
	set	2, a
	ld	(ix + DriveData.stat2), a
	ld	e, HP_ID_AM_DPAT
	ld	d, 0c6h
	jr	x1d53

x1d47:	ld	a, (ix + DriveData.stat2)
	and	~(DT_HP_m | DT_BLANK_m)         ; why not DT_DS_m also?
	set	DT_IBM, a
	ld	(ix + DriveData.stat2), a
	ld	d, 40h
x1d53:	ld	a, (current_unit)
	call	prep_drive_regs
	ld	b, 1
	ld	a, (current_head)
	ld	c, a
	xor	a
	ld	(format_flags), a
	ret

;x1d64:
sel_head_0:
        res	HEAD1, (ix + DriveData.drv)
	res	HEAD1, (iy + YB(current_head))
	jr	sel_head_common

sel_head_1:
        set	HEAD1, (ix + DriveData.drv)
	set	HEAD1, (iy + YB(current_head))

sel_head_common:
        deassert_in_use
	ret

x1d78:	ld	a, (monitor_U6_shdw)
	res	6, a
	or	h
	ld	c, 1dh
	call	update_monitor_leds
	ld	b, 0
	ld	c, 0c6h
	bit	6, h
	jr	nz, x1d8e
	jp	sel_head_0

x1d8e:	bit	3, (ix + DriveData.stat2)
	ret	z
	call	sel_head_1
	or	a
	ret

x1d98:	ld	(ix + DriveData.sector), b
	ld	a, (current_head)
	ld	(ix + DriveData.target_cyl_hd), a
	ld	hl, 0f07fh
	in	a, (switches)
	bit	3, a
	jr	z, x1dae
	set	2, (ix + DriveData.drv)
x1dae:	push	bc
	call	read_sector_jv
	pop	bc
	ld	a, e
	and	l
	ld	e, a
	pop	hl
	bit	4, e
	jp	nz, tsterr_st0
	bit	3, e
	jr	nz, x1dcd
	bit	0, e
	jp	z, tsterr_st1
	bit	0, d
	jp	z, tsterr_st2
	jp	tsterr_st3

x1dcd:	bit	1, e
	jp	nz, tsterr_st4
	bit	5, e
	jr	z, x1de0
	bit	2, (ix + DriveData.drv)
	jp	nz, tsterr
	jp	tsterr_st5

x1de0:	jp	x1fcb

read_test:
	bit	WR_TEST_ACTIVE, (iy + YB(test_flags))
	ret	nz
	in	a, (switches)
	bit	SELFTEST, a
	ret	z                               ; self test SW not on, return
	ld	a, (monitor_U6_shdw)
	res	HEAD, a
	set	1, a                            ; indicate running read-only subtests
	ld	c, 1dh                          ; read test
	call	update_monitor_leds
	in	a, (drv_status)
	bit	READY, a
	jp	z, tsterr_st3
	xor	a
	cp	(ix + DriveData.stat2)
	jr	nz, .skipmtc
	ld	d, a                            ; init DE for media type check
	ld	e, a
	call	get_double_sided_bit            ; e: DriveData.stat2
	call	media_type_check
	jp	nz, tsterr_st0
	ld	(ix + DriveData.stat2), e       ; remember what we learned
	ld	(ix + DriveData.stat2_ds), d
.skipmtc:
        bit	1, (ix + DriveData.stat2)
	jp	nz, tsterr_st1
	ld	a, (current_unit)
	call	prep_drive_regs
	res	7, (iy +14h)
	call	x1e95
	call	seek_to_cylinder_jv
	jr	z, x1e36
x1e30:	jp	nc, tsterr_st2
	jp	c, tsterr_st0
x1e36:	bit	0, (iy + YB(disc_format))
	jr	nz, x1e80
	call	led_data_to_a_c
	set	3, a
	and	0bdh
	call	update_monitor_leds
	ld	b, 0
x1e48:	call	x1d98
	call	x1d32
	jr	nz, x1e48
	bit	3, (ix + DriveData.stat2)
	ret	z
	call	led_data_to_a_c
	or	42h
	call	update_monitor_leds
	set	7, (iy +14h)
	call	x1e95
	call	seek_to_cylinder_jv
	jr	nz, x1e30
	call	led_data_to_a_c
	res	1, a
	call	update_monitor_leds
	ld	b, 0
x1e73:	call	x1d98
	call	x1d32
	jr	nz, x1e73
	res	7, (iy +14h)
	ret

x1e80:	call	led_data_to_a_c
	res	3, a
	and	0bdh
	call	update_monitor_leds
	ld	b, 1
x1e8c:	call	x1d98
	call	x1cd4
	jr	nz, x1e8c
	ret

x1e95:	ld	a, (current_head)
	ld	(ix + DriveData.target_cyl_hd), a
	ret

test_1e9c:
	call	deselect_drives_no_chng_check
	xor	a
	out	(cntl), a
	ld	hl, current_unit
	inc	(hl)
	ld	a, 4
	sub	(hl)
	jr	nz, x1eb5
	ld	(hl), a
	ld	(monitor_U6_shdw), a
	bit	5, (iy + YB(test_flags))
	jr	nz, x1ec3
x1eb5:	ld	hl, seek_recalibrate_test_ptr
	jr	nz, x1ebc
	inc	hl
	inc	hl
x1ebc:	ld	(test_jmptbl_ptr), hl
	ret	nz
	jp	tsterr_st7

x1ec3:	ld	hl, current_head
	res	7, (hl)
	inc	(hl)
	ld	a, 4dh
	sub	(hl)
	jr	z, x1ed7
	res	5, (iy + YB(test_flags))
	set	7, (iy + YB(test_flags))
	ret

x1ed7:	ld	(hl), a
	ret

        ; Clear memory, save contents of leds_shdw
test_1ed9:  pop	bc
	ld	de, (leds_shdw)
	xor	a                       ; Clear RAM
	ld	bc, ramsize
	ld	hl, ramstart
x1ee5:	ld	(hl), a
	inc	hl
	dec	c
	jr	nz, x1ee5
	djnz	x1ee5
	ld	(leds_shdw), de
	jp	done_holdoff

test_1ef3:
	xor	a
	out	(cntl), a
	ld	(leds_shdw), a
	ld	(monitor_U6_shdw), a
	cpl
	out	(monitor_U6), a
	res	0, a
	out	(leds), a
	bit	4, (iy + YB(test_flags))
	ret	z
	in	a, (switches)
	bit	5, a
	jr	nz, x1f19
	bit	4, a
	ret	z
	ld	b, 0fah
	call	delay
	call	delay
x1f19:	ld	b, a
	ld	a, (current_head)
	or	a
	jr	z, x1f2a
	bit	3, (iy + YB(test_flags))
	jr	nz, x1f36
	bit	4, b
	jr	nz, x1f36
x1f2a:	ld	d, 0
	ld	e, (iy + YB(test_flags))
	inc	(iy + YB(test_jmptbl_ptr))
	inc	(iy + YB(test_jmptbl_ptr))
	ret

x1f36:	ld	hl, seek_recalibrate_test_ptr
	ld	(test_jmptbl_ptr), hl
	ret

        ; This bit of code seems to be an odd way to set the sub test number
        ; Check what SP will be at the branches to these entry points.
tsterr_st0:	inc	sp
tsterr_st1:	inc	sp
tsterr_st2:	inc	sp
tsterr_st3:	inc	sp
tsterr_st4:	inc	sp
tsterr_st5:	inc	sp
tsterr_st6:	inc	sp
tsterr_st7:	inc	sp
tsterr:	ld	hl, ramend+6
	scf                             ; clear carry flag for next SBC
	ccf
	sbc	hl, sp                   ; HL = subtest number
	ld	sp, tsterr_sp            ; reset SP so it is usable
	ld	b, 5
x1f51:	sla	l                       ; align subtest number for display (5 bits left)
	djnz	x1f51
	ld	a, (monitor_U6_shdw)
	adc	a, b                     ; B will be zero, so this just adds carry flag
                                        ; which will contain the 4th? bit of subtest#
	set	7, a                     ; Light the ERROR LED
	push	af
	ld	a, (leds_shdw)
	or	l                       ; include subtest portion
	ld	c, a
	pop	af
	call	update_monitor_leds
	xor	a                       ; disable data transfers?
	out	(cntl), a
	bit	4, (iy + YB(test_flags))
	jr	z, x1fc4
	in	a, (switches)
	bit	4, a
	jr	z, x1fa0                 ; self test SW is off
	bit	5, a
	jr	z, test_done             ; the test loop jumper is not installed
	ld	b, 0c8h
	call	delay                   ; delay 200 ms
	ld	a, (monitor_U6_shdw)
	and	30h                     ; turn off ERROR, HEAD and HPFMT LEDs
	ld	(monitor_U6_shdw), a
	res	7, (iy +14h)
	dec	(iy + YB(test_jmptbl_ptr))
	dec	(iy + YB(test_jmptbl_ptr))
	ret

test_done:
        call	deselect_drives_no_chng_check
x1f93:	ld	d, 20h                   ; approx 1/8 s elapsed?
	call	head_unload_timer_check
	in	a, (switches)
	bit	4, a
	jr	nz, x1f93                ; SELF TEST SW pressed
	jr	x1fc4

;run_self_test?:
;stop_self_test?:
x1fa0:	call	deselect_drives_no_chng_check
	ld	a, 0ceh
x1fa5:	push	af
	ld	d, 80h                   ; approx 1/2 s elapsed
	call	head_unload_timer_check
	ld	b, 64h
	call	delay           ; Delay 64 ms
	pop	af
	dec	a
	jr	nz, x1fa5
	in	a, (switches)
	bit	5, a
	jr	z, x1fc4         ; Test LOOP jumper not installed?
	xor	a
	ld	d, a
	ld	e, (iy + YB(test_flags))
	ld	hl, test_1ed9_ptr        ; (last test in test_jump_table)
	jr	x1fc7

x1fc4:	ld	hl, test_1ef3_ptr        ; (second to last test in test_jump_table)
x1fc7:	ld	(test_jmptbl_ptr), hl
	ret

x1fcb:	bit	6, e
	jp	nz, tsterr_st6
	ld	a, (test_flags)
	bit	3, a
	jr	z, x1fde         ; WR TEST SW not set
	ld	a, (data_buffer)
	cp	c               ; C = ??
	jp	nz, tsterr_st7
x1fde:	res	2, (ix + DriveData.drv)
	jp	(hl)

	fillto	01ffeh, 076h

rom_csum_L:	db 0b0h
rom_csum_H:	db 0ffh

	endsection rom

